import { drizzle } from 'drizzle-orm/bun-sql';
import { postgres } from '@agentuity/postgres';
/**
 * Resolves the PostgreSQL client configuration from Drizzle config options.
 *
 * URL priority chain: `connection.url` > `url` > `connectionString` > `process.env.DATABASE_URL`
 *
 * @internal Exported for testing — not part of the public package API.
 */
export function resolvePostgresClientConfig(config) {
    // Clone the connection config to avoid mutating the caller's object
    const clientConfig = config?.connection ? { ...config.connection } : {};
    // Resolve URL using priority chain
    if (!clientConfig.url) {
        if (config?.url) {
            clientConfig.url = config.url;
        }
        else if (config?.connectionString) {
            clientConfig.url = config.connectionString;
        }
        else if (process.env.DATABASE_URL) {
            clientConfig.url = process.env.DATABASE_URL;
        }
    }
    // Add reconnection configuration
    if (config?.reconnect) {
        clientConfig.reconnect = config.reconnect;
    }
    // Add callbacks
    if (config?.onReconnected) {
        clientConfig.onreconnected = config.onReconnected;
    }
    return clientConfig;
}
/**
 * Strips leading whitespace and SQL comments (block and line) from a query string.
 * Returns the remaining query text starting at the first non-comment token.
 */
const LEADING_COMMENTS_RE = /^(?:\s+|\/\*[\s\S]*?\*\/|--[^\n]*\n)*/;
/**
 * Determines whether a SQL query is a non-retryable INSERT statement.
 *
 * Handles two patterns:
 * 1. Direct INSERT: `INSERT INTO ...` (with optional leading comments/whitespace)
 * 2. CTE INSERT: `WITH cte AS (...) INSERT INTO ...` — scans past the WITH clause
 *    by tracking parenthesis depth to skip CTE subexpressions, then checks
 *    if the first top-level DML keyword is INSERT.
 *
 * @see https://github.com/agentuity/sdk/issues/911
 */
function isNonRetryableInsert(query) {
    // Strip leading whitespace and SQL comments
    const stripped = query.replace(LEADING_COMMENTS_RE, '');
    // Fast path: direct INSERT statement
    if (/^INSERT\s/i.test(stripped)) {
        return true;
    }
    // Check for WITH (CTE) prefix
    if (!/^WITH\s/i.test(stripped)) {
        return false;
    }
    // Scan past the CTE clause to find the first top-level DML keyword.
    // We track parenthesis depth so we skip CTE subexpressions like
    // "WITH cte AS (SELECT ... INSERT ...)" without false-matching the
    // INSERT inside the parens.
    let depth = 0;
    let i = 4; // skip past "WITH"
    const len = stripped.length;
    while (i < len) {
        const ch = stripped[i];
        if (ch === '(') {
            depth++;
            i++;
            continue;
        }
        if (ch === ')') {
            depth--;
            i++;
            continue;
        }
        // Only inspect keywords at top level (depth === 0)
        if (depth === 0) {
            // Skip whitespace at top level
            if (ch === ' ' || ch === '\t' || ch === '\n' || ch === '\r') {
                i++;
                continue;
            }
            // Skip commas between CTEs: WITH a AS (...), b AS (...)
            if (ch === ',') {
                i++;
                continue;
            }
            // Check for DML keywords at this position.
            // We look for INSERT, UPDATE, DELETE, or SELECT — the first one
            // we find at top level determines whether this is retryable.
            const rest = stripped.substring(i);
            const dmlMatch = /^(INSERT|UPDATE|DELETE|SELECT)\s/i.exec(rest);
            if (dmlMatch) {
                return dmlMatch[1].toUpperCase() === 'INSERT';
            }
            // Skip over any other word (e.g., CTE names, AS keyword, RECURSIVE)
            // by advancing past alphanumeric/underscore characters
            if (/\w/.test(ch)) {
                while (i < len && /\w/.test(stripped[i])) {
                    i++;
                }
                continue;
            }
        }
        i++;
    }
    return false;
}
/**
 * Creates a dynamic SQL proxy that always delegates to the PostgresClient's
 * current raw connection. This ensures that after automatic reconnection,
 * Drizzle ORM uses the fresh connection instead of a stale reference.
 *
 * The proxy also wraps `unsafe()` calls with the client's retry logic,
 * providing automatic retry on transient connection errors.
 *
 * @internal Exported for testing — not part of the public package API.
 */
export function createResilientSQLProxy(client) {
    return new Proxy({}, {
        get(_target, prop, _receiver) {
            // Always resolve from the CURRENT raw connection (changes after reconnect)
            const raw = client.raw;
            if (prop === 'unsafe') {
                // Wrap unsafe() with retry logic for resilient queries.
                // Returns a thenable that also supports .values() chaining,
                // matching the SQLQuery interface that Drizzle expects:
                //   client.unsafe(query, params)           → Promise<rows>
                //   client.unsafe(query, params).values()   → Promise<rows>
                return (query, params) => {
                    // INSERT statements (including CTE-based) are NOT retried to prevent
                    // duplicate rows. If an INSERT succeeds on the server but the connection
                    // drops before the response, retrying would re-execute it — creating a
                    // duplicate row when the primary key is server-generated.
                    // See: https://github.com/agentuity/sdk/issues/911
                    const isInsert = isNonRetryableInsert(query);
                    if (isInsert) {
                        const makeDirectExecutor = (useValues) => {
                            const currentRaw = client.raw;
                            const q = currentRaw.unsafe(query, params);
                            return useValues ? q.values() : q;
                        };
                        const result = makeDirectExecutor(false);
                        return Object.assign(result, {
                            values: () => makeDirectExecutor(true),
                        });
                    }
                    const makeExecutor = (useValues) => client.executeWithRetry(async () => {
                        // Re-resolve raw inside retry to get post-reconnect instance
                        const currentRaw = client.raw;
                        const q = currentRaw.unsafe(query, params);
                        return useValues ? q.values() : q;
                    });
                    // Return a thenable with .values() to match Bun's SQLQuery interface
                    const result = makeExecutor(false);
                    return Object.assign(result, {
                        values: () => makeExecutor(true),
                    });
                };
            }
            const value = raw[prop];
            if (typeof value === 'function') {
                // Bind to raw so `this` is correct inside begin(), savepoint(), etc.
                return value.bind(raw);
            }
            return value;
        },
    });
}
/**
 * Creates a Drizzle ORM instance with a resilient PostgreSQL connection.
 *
 * This function combines the power of Drizzle ORM with @agentuity/postgres's
 * automatic reconnection capabilities. The underlying connection will
 * automatically reconnect with exponential backoff if the connection is lost.
 *
 * @template TSchema - The Drizzle schema type for type-safe queries
 * @param config - Configuration options for the database connection
 * @returns An object containing the Drizzle instance, underlying client, and close function
 *
 * @example
 * ```typescript
 * import { createPostgresDrizzle } from '@agentuity/drizzle';
 * import * as schema from './schema';
 *
 * // Basic usage with DATABASE_URL
 * const { db, close } = createPostgresDrizzle({ schema });
 *
 * // Query with type safety
 * const users = await db.select().from(schema.users);
 *
 * // Clean up when done
 * await close();
 * ```
 *
 * @example
 * ```typescript
 * // With custom connection configuration
 * const { db, client, close } = createPostgresDrizzle({
 *   connectionString: 'postgres://user:pass@localhost:5432/mydb',
 *   schema,
 *   logger: true,
 *   reconnect: {
 *     maxAttempts: 5,
 *     initialDelayMs: 100,
 *   },
 *   onReconnected: () => console.log('Database reconnected'),
 * });
 *
 * // Access connection stats
 * console.log(client.stats);
 * ```
 */
export function createPostgresDrizzle(config) {
    // Resolve the postgres client configuration
    const clientConfig = resolvePostgresClientConfig(config);
    // Create the postgres client
    const client = postgres(clientConfig);
    // Wait for connection before calling onConnect callback
    // This ensures the callback executes only after the connection is established
    if (config?.onConnect) {
        client.waitForConnection().then(() => {
            config.onConnect();
        });
    }
    // Create a resilient proxy that always delegates to the current raw SQL
    // connection. This ensures that after reconnection, Drizzle automatically
    // uses the new connection instead of the stale one.
    const resilientSQL = createResilientSQLProxy(client);
    // Create Drizzle instance using the resilient proxy instead of a static
    // reference to client.raw, which would become stale after reconnection.
    const db = drizzle({
        client: resilientSQL,
        schema: config?.schema,
        logger: config?.logger,
    });
    // Return the combined interface
    return {
        db,
        client,
        close: async () => {
            await client.close();
        },
    };
}
//# sourceMappingURL=postgres.js.map