import { useCallback, useContext, useEffect, useMemo, useRef, useState } from 'react';
import { buildUrl, EventStreamManager, jsonEqual, } from '@agentuity/frontend';
import { AgentuityContext } from './context';
/**
 * Type-safe EventStream (SSE) hook for connecting to SSE routes.
 *
 * Provides automatic type inference for route outputs based on
 * the SSERouteRegistry generated from your routes.
 *
 * @template TRoute - SSE route key from SSERouteRegistry (e.g., '/events', '/notifications')
 * @template TOutput - Optional type override for SSE event data. When provided, this type
 *   is used instead of the inferred type from the route registry. This is useful for SSE
 *   routes where outputSchema is `never` in the generated types.
 *
 * @example Simple SSE connection
 * ```typescript
 * const { isConnected, data } = useEventStream('/events');
 *
 * // data is fully typed based on route output schema!
 * ```
 *
 * @example SSE with query parameters
 * ```typescript
 * const { isConnected, data } = useEventStream('/notifications', {
 *   query: new URLSearchParams({ userId: '123' })
 * });
 * ```
 *
 * @example SSE with custom output type (when registry has outputSchema: never)
 * ```typescript
 * interface StreamMessage {
 *   type: 'token' | 'complete';
 *   content?: string;
 * }
 *
 * const { isConnected, data } = useEventStream<'/api/search', StreamMessage>('/api/search');
 *
 * // data is typed as StreamMessage | undefined
 * if (data?.type === 'token') {
 *   console.log(data.content);
 * }
 * ```
 */
export function useEventStream(route, options) {
    const context = useContext(AgentuityContext);
    if (!context) {
        throw new Error('useEventStream must be used within a AgentuityProvider');
    }
    const managerRef = useRef(null);
    const [data, setData] = useState();
    const [error, setError] = useState(null);
    const [isError, setIsError] = useState(false);
    const [isConnected, setIsConnected] = useState(false);
    const [readyState, setReadyState] = useState(2); // EventSource.CLOSED = 2
    // Build EventStream URL
    // Track both query object and its string representation to detect mutations.
    // URLSearchParams can be mutated in-place without changing object identity,
    // so we compare the string value to trigger recomputation when params change.
    const queryString = options?.query?.toString();
    const esUrl = useMemo(() => buildUrl(context.baseUrl, route, options?.subpath, options?.query), 
    // biome-ignore lint/correctness/useExhaustiveDependencies: queryString tracks URLSearchParams mutations that options?.query reference wouldn't catch
    [context.baseUrl, route, options?.subpath, options?.query, queryString]);
    // Initialize manager and connect
    useEffect(() => {
        const manager = new EventStreamManager({
            url: esUrl,
            callbacks: {
                onConnect: () => {
                    setIsConnected(true);
                    setError(null);
                    setIsError(false);
                    setReadyState(1); // EventSource.OPEN = 1
                },
                onDisconnect: () => {
                    setIsConnected(false);
                    setReadyState(2); // EventSource.CLOSED = 2
                },
                onError: (err) => {
                    setError(err);
                    setIsError(true);
                },
            },
        });
        // Set message handler with JSON memoization
        manager.setMessageHandler((message) => {
            setData((prev) => (prev !== undefined && jsonEqual(prev, message) ? prev : message));
        });
        manager.connect();
        managerRef.current = manager;
        return () => {
            manager.dispose();
            managerRef.current = null;
        };
    }, [esUrl]);
    // Handle abort signal
    useEffect(() => {
        if (options?.signal) {
            const listener = () => {
                managerRef.current?.close();
            };
            options.signal.addEventListener('abort', listener);
            return () => {
                options.signal?.removeEventListener('abort', listener);
            };
        }
    }, [options?.signal]);
    const reset = useCallback(() => {
        setError(null);
        setIsError(false);
    }, []);
    const close = useCallback(() => {
        managerRef.current?.close();
    }, []);
    return {
        isConnected,
        close,
        data,
        error,
        isError,
        reset,
        readyState,
    };
}
//# sourceMappingURL=eventstream.js.map