/**
 * Symbol used to identify schema types in a minification-safe way.
 * Uses Symbol.for() to ensure the same symbol is used across bundled modules.
 */
export const SCHEMA_KIND = Symbol.for('@agentuity/schema-kind');
/**
 * Error thrown when schema validation fails.
 * Contains detailed information about all validation issues including field paths.
 *
 * @example
 * ```typescript
 * try {
 *   schema.parse(data);
 * } catch (error) {
 *   if (error instanceof ValidationError) {
 *     console.log(error.message);  // Human-readable error
 *     console.log(error.issues);   // Detailed issues array
 *   }
 * }
 * ```
 */
export class ValidationError extends Error {
    /** Array of validation issues with paths and messages */
    issues;
    constructor(issues) {
        const message = issues
            .map((issue) => {
            const path = issue.path
                ? `[${issue.path
                    .map((p) => 
                // eslint-disable-next-line @typescript-eslint/no-explicit-any
                typeof p === 'object' ? p.key : p)
                    .join('.')}]`
                : '';
            return path ? `${path}: ${issue.message}` : issue.message;
        })
            .join('\n');
        super(message);
        this.name = 'ValidationError';
        this.issues = issues;
        // Maintain proper stack trace for where our error was thrown
        if (Error.captureStackTrace) {
            Error.captureStackTrace(this, ValidationError);
        }
    }
    toString() {
        return `${this.name}: ${this.message}`;
    }
}
/**
 * Create a validation issue with an optional field path.
 */
export function createIssue(message, path) {
    return path ? { message, path } : { message };
}
/**
 * Create a successful validation result.
 */
export function success(value) {
    return { value };
}
/**
 * Create a failed validation result.
 */
export function failure(issues) {
    return { issues };
}
/**
 * Create parse and safeParse methods for a schema.
 * @internal
 */
export function createParseMethods() {
    return {
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        parse(value) {
            const result = this['~standard'].validate(value);
            if (result instanceof Promise) {
                throw new Error('Async validation not supported in parse()');
            }
            if (result.issues) {
                throw new ValidationError(result.issues);
            }
            return result.value;
        },
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        safeParse(value) {
            const result = this['~standard'].validate(value);
            if (result instanceof Promise) {
                throw new Error('Async validation not supported in safeParse()');
            }
            if (result.issues) {
                return { success: false, error: new ValidationError(result.issues) };
            }
            return { success: true, data: result.value };
        },
    };
}
//# sourceMappingURL=base.js.map