import type { Logger, SandboxCreateOptions, SandboxInfo, ExecuteOptions as CoreExecuteOptions, Execution, FileToWrite, SandboxRunOptions, SandboxRunResult } from '@agentuity/core';
import type { Readable, Writable } from 'node:stream';
import { type SandboxCreateResponse } from './create';
/**
 * Extended execute options that support piping output to writable streams
 */
export interface ExecuteOptions extends CoreExecuteOptions {
    /**
     * Pipe stdout/stderr to writable streams (e.g., process.stdout)
     */
    pipe?: {
        stdout?: Writable;
        stderr?: Writable;
    };
}
export interface SandboxClientOptions {
    /**
     * API key for authentication.
     * Defaults to process.env.AGENTUITY_SDK_KEY || process.env.AGENTUITY_CLI_KEY
     */
    apiKey?: string;
    /**
     * Base URL for the sandbox API.
     * Defaults to process.env.AGENTUITY_SANDBOX_URL ||
     *   process.env.AGENTUITY_CATALYST_URL ||
     *   process.env.AGENTUITY_TRANSPORT_URL ||
     *   regional catalyst URL
     */
    url?: string;
    /**
     * Organization ID for multi-tenant operations
     */
    orgId?: string;
    /**
     * Custom logger instance
     */
    logger?: Logger;
}
/**
 * I/O options for one-shot sandbox execution via run()
 */
export interface SandboxClientRunIO {
    /**
     * AbortSignal to cancel the execution
     */
    signal?: AbortSignal;
    /**
     * Readable stream for stdin input
     */
    stdin?: Readable;
    /**
     * Writable stream for stdout output
     */
    stdout?: Writable;
    /**
     * Writable stream for stderr output
     */
    stderr?: Writable;
    /**
     * Optional logger override for this run
     */
    logger?: Logger;
}
/**
 * A sandbox instance returned by SandboxClient.create()
 */
export interface SandboxInstance {
    /**
     * Unique sandbox identifier
     */
    id: string;
    /**
     * Sandbox status at creation time
     */
    status: SandboxCreateResponse['status'];
    /**
     * URL to stream stdout output
     */
    stdoutStreamUrl?: string;
    /**
     * URL to stream stderr output
     */
    stderrStreamUrl?: string;
    /**
     * Execute a command in the sandbox
     */
    execute(options: ExecuteOptions): Promise<Execution>;
    /**
     * Write files to the sandbox workspace
     */
    writeFiles(files: FileToWrite[]): Promise<number>;
    /**
     * Read a file from the sandbox workspace
     */
    readFile(path: string): Promise<ReadableStream<Uint8Array>>;
    /**
     * Get current sandbox information
     */
    get(): Promise<SandboxInfo>;
    /**
     * Destroy the sandbox and release all resources
     */
    destroy(): Promise<void>;
}
/**
 * Convenience client for sandbox operations.
 *
 * @example
 * ```typescript
 * // Interactive sandbox usage
 * const client = new SandboxClient();
 * const sandbox = await client.create();
 * const result = await sandbox.execute({ command: ['echo', 'hello'] });
 * await sandbox.destroy();
 *
 * // One-shot execution with streaming
 * const result = await client.run(
 *   { command: { exec: ['bun', 'run', 'script.ts'] } },
 *   { stdout: process.stdout, stderr: process.stderr }
 * );
 * ```
 */
export declare class SandboxClient {
    #private;
    constructor(options?: SandboxClientOptions);
    /**
     * Run a one-shot command in a new sandbox (creates, executes, destroys)
     *
     * This is a high-level convenience method that handles the full lifecycle:
     * creating a sandbox, streaming I/O, polling for completion, and cleanup.
     *
     * @param options - Execution options including command and configuration
     * @param io - Optional I/O streams and abort signal
     * @returns The run result including exit code and duration
     * @throws {Error} If stdin is provided without an API key
     *
     * @example
     * ```typescript
     * const client = new SandboxClient();
     * const result = await client.run(
     *   { command: { exec: ['bun', 'run', 'script.ts'] } },
     *   { stdout: process.stdout, stderr: process.stderr }
     * );
     * console.log('Exit code:', result.exitCode);
     * ```
     */
    run(options: SandboxRunOptions, io?: SandboxClientRunIO): Promise<SandboxRunResult>;
    /**
     * Create a new sandbox instance
     *
     * @param options - Optional sandbox configuration
     * @returns A sandbox instance with execute and destroy methods
     */
    create(options?: SandboxCreateOptions): Promise<SandboxInstance>;
    /**
     * Get sandbox information by ID
     *
     * @param sandboxId - The sandbox ID
     * @returns Sandbox information
     */
    get(sandboxId: string): Promise<SandboxInfo>;
    /**
     * Destroy a sandbox by ID
     *
     * @param sandboxId - The sandbox ID to destroy
     */
    destroy(sandboxId: string): Promise<void>;
    /**
     * Write files to a sandbox workspace
     *
     * @param sandboxId - The sandbox ID
     * @param files - Array of files to write with path and content
     * @param signal - Optional AbortSignal to cancel the operation
     * @returns The number of files written
     */
    writeFiles(sandboxId: string, files: FileToWrite[], signal?: AbortSignal): Promise<number>;
    /**
     * Read a file from a sandbox workspace
     *
     * @param sandboxId - The sandbox ID
     * @param path - Path to the file relative to the sandbox workspace
     * @param signal - Optional AbortSignal to cancel the operation
     * @returns A ReadableStream of the file contents
     */
    readFile(sandboxId: string, path: string, signal?: AbortSignal): Promise<ReadableStream<Uint8Array>>;
}
//# sourceMappingURL=client.d.ts.map