import { upgradeWebSocket } from 'hono/bun';
import { context as otelContext, ROOT_CONTEXT } from '@opentelemetry/api';
import { getAgentAsyncLocalStorage } from '../_context';
/**
 * Creates a WebSocket middleware for handling WebSocket connections.
 *
 * Use with router.get() to create a WebSocket endpoint:
 *
 * @example
 * ```typescript
 * import { createRouter, websocket } from '@agentuity/runtime';
 *
 * const router = createRouter();
 *
 * router.get('/ws', websocket((c, ws) => {
 *   ws.onOpen(() => {
 *     c.var.logger.info('WebSocket opened');
 *     ws.send('Welcome!');
 *   });
 *
 *   ws.onMessage((event) => {
 *     c.var.logger.info('Received:', event.data);
 *     ws.send('Echo: ' + event.data);
 *   });
 *
 *   ws.onClose(() => {
 *     c.var.logger.info('WebSocket closed');
 *   });
 * }));
 * ```
 *
 * @param handler - Handler function receiving context and WebSocket connection
 * @returns Hono middleware handler for WebSocket upgrade
 */
export function websocket(handler) {
    const wsHandler = upgradeWebSocket((c) => {
        let openHandler;
        let messageHandler;
        let closeHandler;
        let initialized = false;
        const asyncLocalStorage = getAgentAsyncLocalStorage();
        const capturedContext = asyncLocalStorage.getStore();
        const wsConnection = {
            onOpen: (h) => {
                openHandler = h;
            },
            onMessage: (h) => {
                messageHandler = h;
            },
            onClose: (h) => {
                closeHandler = h;
            },
            send: (_data) => {
                // This will be bound to the actual ws in the handlers
            },
        };
        // IMPORTANT: We run in ROOT_CONTEXT (no active OTEL span) to avoid a Bun bug
        // where OTEL-instrumented fetch conflicts with streaming responses.
        // See: https://github.com/agentuity/sdk/issues/471
        // See: https://github.com/oven-sh/bun/issues/24766
        const runHandler = () => {
            otelContext.with(ROOT_CONTEXT, () => {
                if (capturedContext) {
                    asyncLocalStorage.run(capturedContext, () => handler(c, wsConnection));
                }
                else {
                    handler(c, wsConnection);
                }
            });
            initialized = true;
        };
        runHandler();
        return {
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            onOpen: async (event, ws) => {
                try {
                    wsConnection.send = (data) => ws.send(data);
                    if (openHandler) {
                        const h = openHandler;
                        await otelContext.with(ROOT_CONTEXT, async () => {
                            if (capturedContext) {
                                await asyncLocalStorage.run(capturedContext, () => h(event));
                            }
                            else {
                                await h(event);
                            }
                        });
                    }
                }
                catch (err) {
                    c.var.logger?.error('WebSocket onOpen error:', err);
                    throw err;
                }
            },
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            onMessage: async (event, ws) => {
                try {
                    if (!initialized) {
                        wsConnection.send = (data) => ws.send(data);
                        runHandler();
                    }
                    if (messageHandler) {
                        const h = messageHandler;
                        await otelContext.with(ROOT_CONTEXT, async () => {
                            if (capturedContext) {
                                await asyncLocalStorage.run(capturedContext, () => h(event));
                            }
                            else {
                                await h(event);
                            }
                        });
                    }
                }
                catch (err) {
                    c.var.logger?.error('WebSocket onMessage error:', err);
                    throw err;
                }
            },
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            onClose: async (event, _ws) => {
                try {
                    if (closeHandler) {
                        const h = closeHandler;
                        await otelContext.with(ROOT_CONTEXT, async () => {
                            if (capturedContext) {
                                await asyncLocalStorage.run(capturedContext, () => h(event));
                            }
                            else {
                                await h(event);
                            }
                        });
                    }
                }
                catch (err) {
                    c.var.logger?.error('WebSocket onClose error:', err);
                }
            },
        };
    });
    const middleware = (c, next) => wsHandler(c, next);
    return middleware;
}
//# sourceMappingURL=websocket.js.map