import type { Logger } from './types';
/**
 * Standard exit codes for the CLI
 */
export declare enum ExitCode {
    SUCCESS = 0,
    GENERAL_ERROR = 1,
    VALIDATION_ERROR = 2,
    AUTH_ERROR = 3,
    NOT_FOUND = 4,
    PERMISSION_ERROR = 5,
    NETWORK_ERROR = 6,
    FILE_ERROR = 7,
    USER_CANCELLED = 8,
    BUILD_FAILED = 9,
    SECURITY_ERROR = 10
}
/**
 * Standard error codes for the CLI
 */
export declare enum ErrorCode {
    VALIDATION_FAILED = "VALIDATION_FAILED",
    MISSING_ARGUMENT = "MISSING_ARGUMENT",
    INVALID_ARGUMENT = "INVALID_ARGUMENT",
    INVALID_OPTION = "INVALID_OPTION",
    UNKNOWN_COMMAND = "UNKNOWN_COMMAND",
    AUTH_REQUIRED = "AUTH_REQUIRED",
    AUTH_FAILED = "AUTH_FAILED",
    AUTH_EXPIRED = "AUTH_EXPIRED",
    PERMISSION_DENIED = "PERMISSION_DENIED",
    PROJECT_NOT_FOUND = "PROJECT_NOT_FOUND",
    PROJECT_INVALID = "PROJECT_INVALID",
    CONFIG_NOT_FOUND = "CONFIG_NOT_FOUND",
    CONFIG_INVALID = "CONFIG_INVALID",
    RESOURCE_NOT_FOUND = "RESOURCE_NOT_FOUND",
    RESOURCE_ALREADY_EXISTS = "RESOURCE_ALREADY_EXISTS",
    RESOURCE_CONFLICT = "RESOURCE_CONFLICT",
    ORG_REQUIRED = "ORG_REQUIRED",
    ORG_NOT_FOUND = "ORG_NOT_FOUND",
    REGION_REQUIRED = "REGION_REQUIRED",
    REGION_NOT_FOUND = "REGION_NOT_FOUND",
    NO_REGIONS_AVAILABLE = "NO_REGIONS_AVAILABLE",
    NETWORK_ERROR = "NETWORK_ERROR",
    API_ERROR = "API_ERROR",
    TIMEOUT = "TIMEOUT",
    FILE_NOT_FOUND = "FILE_NOT_FOUND",
    FILE_READ_ERROR = "FILE_READ_ERROR",
    FILE_WRITE_ERROR = "FILE_WRITE_ERROR",
    DIRECTORY_NOT_FOUND = "DIRECTORY_NOT_FOUND",
    RUNTIME_ERROR = "RUNTIME_ERROR",
    INTERNAL_ERROR = "INTERNAL_ERROR",
    NOT_IMPLEMENTED = "NOT_IMPLEMENTED",
    USER_CANCELLED = "USER_CANCELLED",
    BUILD_FAILED = "BUILD_FAILED",
    INTEGRATION_FAILED = "INTEGRATION_FAILED",
    MALWARE_DETECTED = "MALWARE_DETECTED",
    UPGRADE_REQUIRED = "UPGRADE_REQUIRED",
    PAYMENT_REQUIRED = "PAYMENT_REQUIRED"
}
/**
 * Map error codes to exit codes
 */
export declare function getExitCode(errorCode: ErrorCode): ExitCode;
/**
 * Structured error information
 */
export interface StructuredError {
    code: ErrorCode;
    message: string;
    details?: Record<string, unknown>;
    suggestions?: string[];
    exitCode?: ExitCode;
}
/**
 * Format error in JSON structure
 */
export declare function formatErrorJSON(error: StructuredError): string;
/**
 * Format error for human-readable output
 */
export declare function formatErrorHuman(error: StructuredError): string;
/**
 * Exit the process with a structured error
 */
export declare function exitWithError(error: StructuredError, logger: Logger, errorFormat?: 'json' | 'text'): never;
/**
 * Create a structured error from a simple message and code
 */
export declare function createError(code: ErrorCode, message: string, details?: Record<string, unknown>, suggestions?: string[]): StructuredError;
//# sourceMappingURL=errors.d.ts.map