import { z } from 'zod';
import { APIClient } from '../api';
import type { FileToWrite } from '@agentuity/core';
export declare const FileToWriteSchema: z.ZodObject<{
    path: z.ZodString;
    content: z.ZodString;
}, z.core.$strip>;
export declare const WriteFilesRequestSchema: z.ZodObject<{
    files: z.ZodArray<z.ZodObject<{
        path: z.ZodString;
        content: z.ZodString;
    }, z.core.$strip>>;
}, z.core.$strip>;
export declare const WriteFilesDataSchema: z.ZodObject<{
    filesWritten: z.ZodNumber;
}, z.core.$strip>;
export declare const WriteFilesResponseSchema: z.ZodDiscriminatedUnion<[z.ZodObject<{
    success: z.ZodLiteral<false>;
    message: z.ZodString;
}, z.core.$strip>, z.ZodObject<{
    success: z.ZodLiteral<true>;
    data: z.ZodOptional<z.ZodObject<{
        filesWritten: z.ZodNumber;
    }, z.core.$strip>>;
    filesWritten: z.ZodOptional<z.ZodNumber>;
}, z.core.$strip>], "success">;
export interface WriteFilesParams {
    sandboxId: string;
    files: FileToWrite[];
    orgId?: string;
    signal?: AbortSignal;
}
export interface WriteFilesResult {
    filesWritten: number;
}
/**
 * Writes files to a sandbox workspace.
 *
 * @param client - The API client to use for the request
 * @param params - Parameters including sandbox ID and files to write
 * @returns The result including number of files written
 * @throws {SandboxResponseError} If the write request fails
 */
export declare function sandboxWriteFiles(client: APIClient, params: WriteFilesParams): Promise<WriteFilesResult>;
export interface ReadFileParams {
    sandboxId: string;
    path: string;
    orgId?: string;
    signal?: AbortSignal;
}
/**
 * Reads a file from a sandbox workspace.
 *
 * @param client - The API client to use for the request
 * @param params - Parameters including sandbox ID and file path
 * @returns A ReadableStream of the file contents
 * @throws {SandboxResponseError} If the read request fails
 */
export declare function sandboxReadFile(client: APIClient, params: ReadFileParams): Promise<ReadableStream<Uint8Array>>;
export declare const MkDirRequestSchema: z.ZodObject<{
    path: z.ZodString;
    recursive: z.ZodOptional<z.ZodBoolean>;
}, z.core.$strip>;
export declare const MkDirResponseSchema: z.ZodDiscriminatedUnion<[z.ZodObject<{
    success: z.ZodLiteral<false>;
    message: z.ZodString;
}, z.core.$strip>, z.ZodObject<{
    success: z.ZodLiteral<true>;
}, z.core.$strip>], "success">;
export interface MkDirParams {
    sandboxId: string;
    path: string;
    recursive?: boolean;
    orgId?: string;
    signal?: AbortSignal;
}
/**
 * Creates a directory in a sandbox workspace.
 *
 * @param client - The API client to use for the request
 * @param params - Parameters including sandbox ID, path, and recursive flag
 * @throws {SandboxResponseError} If the mkdir request fails
 */
export declare function sandboxMkDir(client: APIClient, params: MkDirParams): Promise<void>;
export declare const RmDirRequestSchema: z.ZodObject<{
    path: z.ZodString;
    recursive: z.ZodOptional<z.ZodBoolean>;
}, z.core.$strip>;
export declare const RmDirResponseSchema: z.ZodDiscriminatedUnion<[z.ZodObject<{
    success: z.ZodLiteral<false>;
    message: z.ZodString;
}, z.core.$strip>, z.ZodObject<{
    success: z.ZodLiteral<true>;
}, z.core.$strip>], "success">;
export interface RmDirParams {
    sandboxId: string;
    path: string;
    recursive?: boolean;
    orgId?: string;
    signal?: AbortSignal;
}
/**
 * Removes a directory from a sandbox workspace.
 *
 * @param client - The API client to use for the request
 * @param params - Parameters including sandbox ID, path, and recursive flag
 * @throws {SandboxResponseError} If the rmdir request fails
 */
export declare function sandboxRmDir(client: APIClient, params: RmDirParams): Promise<void>;
export declare const RmFileRequestSchema: z.ZodObject<{
    path: z.ZodString;
}, z.core.$strip>;
export declare const RmFileResponseSchema: z.ZodDiscriminatedUnion<[z.ZodObject<{
    success: z.ZodLiteral<false>;
    message: z.ZodString;
}, z.core.$strip>, z.ZodObject<{
    success: z.ZodLiteral<true>;
}, z.core.$strip>], "success">;
export interface RmFileParams {
    sandboxId: string;
    path: string;
    orgId?: string;
    signal?: AbortSignal;
}
/**
 * Removes a file from a sandbox workspace.
 *
 * @param client - The API client to use for the request
 * @param params - Parameters including sandbox ID and path
 * @throws {SandboxResponseError} If the rm request fails
 */
export declare function sandboxRmFile(client: APIClient, params: RmFileParams): Promise<void>;
export declare const FileInfoSchema: z.ZodObject<{
    path: z.ZodString;
    size: z.ZodNumber;
    isDir: z.ZodBoolean;
    mode: z.ZodString;
    modTime: z.ZodString;
}, z.core.$strip>;
export declare const ListFilesDataSchema: z.ZodObject<{
    files: z.ZodArray<z.ZodObject<{
        path: z.ZodString;
        size: z.ZodNumber;
        isDir: z.ZodBoolean;
        mode: z.ZodString;
        modTime: z.ZodString;
    }, z.core.$strip>>;
}, z.core.$strip>;
export declare const ListFilesResponseSchema: z.ZodDiscriminatedUnion<[z.ZodObject<{
    success: z.ZodLiteral<false>;
    message: z.ZodString;
}, z.core.$strip>, z.ZodObject<{
    success: z.ZodLiteral<true>;
    data: z.ZodObject<{
        files: z.ZodArray<z.ZodObject<{
            path: z.ZodString;
            size: z.ZodNumber;
            isDir: z.ZodBoolean;
            mode: z.ZodString;
            modTime: z.ZodString;
        }, z.core.$strip>>;
    }, z.core.$strip>;
}, z.core.$strip>], "success">;
export interface FileInfo {
    path: string;
    size: number;
    isDir: boolean;
    mode: string;
    modTime: string;
}
export interface ListFilesParams {
    sandboxId: string;
    path?: string;
    orgId?: string;
    signal?: AbortSignal;
}
export interface ListFilesResult {
    files: FileInfo[];
}
/**
 * Lists files in a sandbox workspace directory.
 *
 * @param client - The API client to use for the request
 * @param params - Parameters including sandbox ID and optional path
 * @returns The list of files and directories
 * @throws {SandboxResponseError} If the list request fails
 */
export declare function sandboxListFiles(client: APIClient, params: ListFilesParams): Promise<ListFilesResult>;
export type ArchiveFormat = 'zip' | 'tar.gz';
export interface DownloadArchiveParams {
    sandboxId: string;
    path?: string;
    format?: ArchiveFormat;
    orgId?: string;
    signal?: AbortSignal;
}
/**
 * Downloads files from a sandbox as a compressed archive.
 *
 * @param client - The API client to use for the request
 * @param params - Parameters including sandbox ID, path, and format
 * @returns A ReadableStream of the archive contents
 * @throws {SandboxResponseError} If the download request fails
 */
export declare function sandboxDownloadArchive(client: APIClient, params: DownloadArchiveParams): Promise<ReadableStream<Uint8Array>>;
export interface UploadArchiveParams {
    sandboxId: string;
    archive: Uint8Array | ArrayBuffer | ReadableStream<Uint8Array>;
    path?: string;
    format?: ArchiveFormat | '';
    orgId?: string;
    signal?: AbortSignal;
}
export declare const UploadArchiveResponseSchema: z.ZodDiscriminatedUnion<[z.ZodObject<{
    success: z.ZodLiteral<false>;
    message: z.ZodString;
}, z.core.$strip>, z.ZodObject<{
    success: z.ZodLiteral<true>;
}, z.core.$strip>], "success">;
/**
 * Uploads a compressed archive to a sandbox and extracts it.
 *
 * @param client - The API client to use for the request
 * @param params - Parameters including sandbox ID, archive data, path, and optional format
 * @throws {SandboxResponseError} If the upload request fails
 */
export declare function sandboxUploadArchive(client: APIClient, params: UploadArchiveParams): Promise<void>;
export declare const SetEnvRequestSchema: z.ZodObject<{
    env: z.ZodRecord<z.ZodString, z.ZodNullable<z.ZodString>>;
}, z.core.$strip>;
export declare const SetEnvDataSchema: z.ZodObject<{
    env: z.ZodRecord<z.ZodString, z.ZodString>;
}, z.core.$strip>;
export declare const SetEnvResponseSchema: z.ZodDiscriminatedUnion<[z.ZodObject<{
    success: z.ZodLiteral<false>;
    message: z.ZodString;
}, z.core.$strip>, z.ZodObject<{
    success: z.ZodLiteral<true>;
    data: z.ZodObject<{
        env: z.ZodRecord<z.ZodString, z.ZodString>;
    }, z.core.$strip>;
}, z.core.$strip>], "success">;
export interface SetEnvParams {
    sandboxId: string;
    env: Record<string, string | null>;
    orgId?: string;
    signal?: AbortSignal;
}
export interface SetEnvResult {
    env: Record<string, string>;
}
/**
 * Sets environment variables on a sandbox. Pass null to delete a variable.
 *
 * @param client - The API client to use for the request
 * @param params - Parameters including sandbox ID and env key/value pairs
 * @returns The current environment variables after the update
 * @throws {SandboxResponseError} If the request fails
 */
export declare function sandboxSetEnv(client: APIClient, params: SetEnvParams): Promise<SetEnvResult>;
//# sourceMappingURL=files.d.ts.map