import { SQL } from 'bun';
import type { ReconnectConfig } from './types';
/**
 * Patches Bun's native SQL class to add automatic reconnection support.
 *
 * This modifies the global `Bun.SQL` prototype to intercept connection close
 * events and automatically attempt reconnection with exponential backoff.
 *
 * **Note:** This is a global modification that affects all SQL instances created
 * after calling this function. Use with caution in shared environments.
 *
 * @param config - Optional configuration for reconnection behavior
 *
 * @example
 * ```typescript
 * import { patchBunSQL, SQL } from '@agentuity/postgres';
 *
 * // Patch with default settings
 * patchBunSQL();
 *
 * // Or with custom configuration
 * patchBunSQL({
 *   reconnect: {
 *     maxAttempts: 5,
 *     initialDelayMs: 200,
 *   },
 *   onreconnect: (attempt) => console.log(`Reconnecting... attempt ${attempt}`),
 *   onreconnected: () => console.log('Reconnected!'),
 * });
 *
 * // Now use Bun.SQL normally - it will auto-reconnect
 * const sql = new SQL({ url: process.env.DATABASE_URL });
 * const users = await sql`SELECT * FROM users`;
 * ```
 */
export declare function patchBunSQL(config?: {
    reconnect?: ReconnectConfig;
    onreconnect?: (attempt: number) => void;
    onreconnected?: () => void;
    onreconnectfailed?: (error: Error) => void;
}): void;
/**
 * Returns whether Bun.SQL has been patched.
 */
export declare function isPatched(): boolean;
/**
 * Resets the patch state (mainly for testing).
 * @internal
 */
export declare function _resetPatch(): void;
/**
 * Re-export of Bun's SQL class.
 *
 * When using the patched version, import SQL from this module instead of 'bun':
 *
 * @example
 * ```typescript
 * import { patchBunSQL, SQL } from '@agentuity/postgres';
 *
 * patchBunSQL();
 * const sql = new SQL({ url: process.env.DATABASE_URL });
 * ```
 */
export { SQL };
//# sourceMappingURL=patch.d.ts.map