import type { Schema, Infer } from '../base';
import { SCHEMA_KIND } from '../base';
import { OptionalSchema } from '../utils/optional';
type ObjectShape = Record<string, Schema<any, any>>;
type IsOptional<T> = T extends OptionalSchema<any> ? true : false;
type RequiredKeys<T extends ObjectShape> = {
    [K in keyof T]: IsOptional<T[K]> extends true ? never : K;
}[keyof T];
type OptionalKeys<T extends ObjectShape> = {
    [K in keyof T]: IsOptional<T[K]> extends true ? K : never;
}[keyof T];
type InferObjectShape<T extends ObjectShape> = {
    [K in RequiredKeys<T>]: Infer<T[K]>;
} & {
    [K in OptionalKeys<T>]?: Infer<T[K]>;
};
type PartialShape<T extends ObjectShape> = {
    [K in keyof T]: OptionalSchema<T[K]>;
};
type PickShape<T extends ObjectShape, K extends keyof T> = Pick<T, K>;
type OmitShape<T extends ObjectShape, K extends keyof T> = Omit<T, K>;
type ExtendShape<T extends ObjectShape, U extends ObjectShape> = Omit<T, keyof U> & U;
/**
 * Schema for validating objects with typed properties.
 * Validates each property according to its schema and collects all validation errors.
 *
 * @template T - The object shape definition
 *
 * @example
 * ```typescript
 * const userSchema = s.object({
 *   name: s.string(),
 *   age: s.number(),
 *   email: s.string()
 * });
 *
 * const user = userSchema.parse({
 *   name: 'John',
 *   age: 30,
 *   email: 'john@example.com'
 * });
 * ```
 */
export declare class ObjectSchema<T extends ObjectShape> implements Schema<InferObjectShape<T>, InferObjectShape<T>> {
    private shape;
    readonly [SCHEMA_KIND] = "ObjectSchema";
    description?: string;
    private parseMethods;
    constructor(shape: T);
    readonly '~standard': {
        version: 1;
        vendor: string;
        validate: (value: unknown) => import("@agentuity/core").StandardSchemaV1.FailureResult | import("@agentuity/core").StandardSchemaV1.SuccessResult<InferObjectShape<T>>;
        types: {
            input: InferObjectShape<T>;
            output: InferObjectShape<T>;
        };
    };
    describe(description: string): this;
    optional(): OptionalSchema<this>;
    nullable(): import("..").NullableSchema<this>;
    /**
     * Create a new object schema with only the specified keys.
     * Similar to TypeScript's Pick<T, K> utility type.
     *
     * @param keys - Array of keys to pick from the schema
     *
     * @example
     * ```typescript
     * const userSchema = s.object({ name: s.string(), age: s.number(), email: s.string() });
     * const nameOnlySchema = userSchema.pick(['name']);
     * // { name: string }
     * ```
     */
    pick<K extends keyof T & string>(keys: readonly K[]): ObjectSchema<PickShape<T, K>>;
    /**
     * Create a new object schema without the specified keys.
     * Similar to TypeScript's Omit<T, K> utility type.
     *
     * @param keys - Array of keys to omit from the schema
     *
     * @example
     * ```typescript
     * const userSchema = s.object({ name: s.string(), age: s.number(), password: s.string() });
     * const publicUserSchema = userSchema.omit(['password']);
     * // { name: string; age: number }
     * ```
     */
    omit<K extends keyof T & string>(keys: readonly K[]): ObjectSchema<OmitShape<T, K>>;
    /**
     * Create a new object schema where all properties are optional.
     * Similar to TypeScript's Partial<T> utility type.
     *
     * @example
     * ```typescript
     * const userSchema = s.object({ name: s.string(), age: s.number() });
     * const partialUserSchema = userSchema.partial();
     * // { name?: string; age?: number }
     * ```
     */
    partial(): ObjectSchema<PartialShape<T>>;
    /**
     * Create a new object schema by extending this schema with additional properties.
     * Properties in the extension override properties in the base schema.
     * Similar to TypeScript's intersection types with override semantics.
     *
     * @param extension - Object shape to extend with
     *
     * @example
     * ```typescript
     * const userSchema = s.object({ name: s.string(), age: s.number() });
     * const adminSchema = userSchema.extend({ role: s.literal('admin'), permissions: s.array(s.string()) });
     * // { name: string; age: number; role: 'admin'; permissions: string[] }
     * ```
     */
    extend<U extends ObjectShape>(extension: U): ObjectSchema<ExtendShape<T, U>>;
    parse: (this: Schema<any, InferObjectShape<T>>, value: unknown) => InferObjectShape<T>;
    safeParse: (this: Schema<any, InferObjectShape<T>>, value: unknown) => import("..").SafeParseResult<InferObjectShape<T>>;
}
/**
 * Create an object schema with typed properties.
 *
 * @param shape - Object defining the schema for each property
 *
 * @example
 * ```typescript
 * const userSchema = s.object({
 *   name: s.string().describe('Full name'),
 *   age: s.number().describe('Age in years'),
 *   email: s.optional(s.string())
 * });
 *
 * type User = s.infer<typeof userSchema>;
 * const user = userSchema.parse(data);
 * ```
 */
export declare function object<T extends ObjectShape>(shape: T): ObjectSchema<T>;
export {};
//# sourceMappingURL=object.d.ts.map