import type { Context, MiddlewareHandler } from 'hono';
import type { Env } from '../app';
import { WebRTCRoomManager, type WebRTCOptions } from '../webrtc-signaling';
export type { WebRTCOptions };
/**
 * Handler function for WebRTC signaling connections.
 * Receives the Hono context and WebRTCRoomManager.
 */
export type WebRTCHandler<E extends Env = Env> = (c: Context<E>, roomManager: WebRTCRoomManager) => void | Promise<void>;
/**
 * Creates a WebRTC signaling middleware for peer-to-peer communication.
 *
 * This middleware sets up a WebSocket-based signaling server that handles:
 * - Room membership and peer discovery
 * - SDP offer/answer relay
 * - ICE candidate relay
 *
 * Use with router.get() to create a WebRTC signaling endpoint:
 *
 * @example
 * ```typescript
 * import { createRouter, webrtc } from '@agentuity/runtime';
 *
 * const router = createRouter();
 *
 * // Basic signaling endpoint
 * router.get('/call/signal', webrtc());
 *
 * // With options
 * router.get('/call/signal', webrtc({ maxPeers: 4 }));
 *
 * // With callbacks for monitoring
 * router.get('/call/signal', webrtc({
 *   maxPeers: 2,
 *   callbacks: {
 *     onRoomCreated: (roomId) => console.log(`Room ${roomId} created`),
 *     onPeerJoin: (peerId, roomId) => console.log(`${peerId} joined ${roomId}`),
 *     onPeerLeave: (peerId, roomId, reason) => {
 *       console.log(`${peerId} left ${roomId}: ${reason}`);
 *     },
 *   },
 * }));
 * ```
 *
 * @param options - Configuration options for WebRTC signaling
 * @returns Hono middleware handler for WebSocket upgrade
 */
export declare function webrtc<E extends Env = Env>(options?: WebRTCOptions): MiddlewareHandler<E>;
//# sourceMappingURL=webrtc.d.ts.map