import type { Context, Handler } from 'hono';
import type { Schema } from '@agentuity/schema';
import type { Env } from '../app';
/**
 * Context variable key for stream completion promise.
 * Used by middleware to defer session/thread saving until stream completes.
 * @internal
 */
export declare const STREAM_DONE_PROMISE_KEY = "_streamDonePromise";
/**
 * Context variable key to indicate this is a streaming response.
 * @internal
 */
export declare const IS_STREAMING_RESPONSE_KEY = "_isStreamingResponse";
/**
 * SSE message format for Server-Sent Events.
 */
export interface SSEMessage {
    data: string;
    event?: string;
    id?: string;
    retry?: number;
}
/**
 * SSE stream interface for writing Server-Sent Events.
 */
export interface SSEStream {
    /**
     * Write a simple value as SSE data.
     * Strings, numbers, and booleans are converted to string data.
     * Objects are passed through as SSE message format.
     */
    write: (data: string | number | boolean | SSEMessage) => Promise<void>;
    /**
     * Write a properly formatted SSE message.
     */
    writeSSE: (message: SSEMessage) => Promise<void>;
    /**
     * Register a callback for when the client aborts the connection.
     */
    onAbort: (callback: () => void) => void;
    /**
     * Close the SSE stream.
     */
    close: () => void;
}
/**
 * Handler function for SSE connections.
 * Receives the Hono context and SSE stream with a flattened signature.
 */
export type SSEHandler<E extends Env = Env> = (c: Context<E>, stream: SSEStream) => void | Promise<void>;
/**
 * Options for configuring SSE middleware.
 *
 * @template TOutput - The type of data that will be sent through the SSE stream.
 *   This is used for type inference in generated route registries and does not
 *   perform runtime validation (SSE data is serialized via JSON.stringify).
 */
export interface SSEOptions<TOutput = unknown> {
    /**
     * Schema defining the output type for SSE events.
     *
     * This schema is used for:
     * - Type inference in generated `routes.ts` registry
     * - Automatic typing of `useEventStream` hook's `data` property
     *
     * The schema is NOT used for runtime validation - SSE messages are sent
     * as-is through the stream. Use this for TypeScript type safety only.
     *
     * @example
     * ```typescript
     * import { s } from '@agentuity/schema';
     *
     * const StreamEventSchema = s.object({
     *   type: s.enum(['token', 'complete', 'error']),
     *   content: s.optional(s.string()),
     * });
     *
     * router.get('/stream', sse({ output: StreamEventSchema }, async (c, stream) => {
     *   await stream.writeSSE({ data: JSON.stringify({ type: 'token', content: 'Hello' }) });
     *   await stream.writeSSE({ data: JSON.stringify({ type: 'complete' }) });
     *   stream.close();
     * }));
     * ```
     */
    output: Schema<TOutput, TOutput>;
}
/**
 * Creates an SSE (Server-Sent Events) middleware for streaming updates to clients.
 *
 * This implementation uses Hono's stream() helper instead of streamSSE() to ensure
 * compatibility with async operations that consume ReadableStreams internally
 * (like AI SDK's generateText/generateObject). The stream() helper uses a fire-and-forget
 * pattern that avoids "ReadableStream has already been used" errors.
 *
 * Use with router.get() to create an SSE endpoint:
 *
 * @example Basic SSE without typed output
 * ```typescript
 * import { createRouter, sse } from '@agentuity/runtime';
 *
 * const router = createRouter();
 *
 * router.get('/events', sse((c, stream) => {
 *   let count = 0;
 *   const interval = setInterval(() => {
 *     stream.writeSSE({
 *       data: `Event ${++count}`,
 *       event: 'update'
 *     });
 *     if (count >= 10) {
 *       clearInterval(interval);
 *       stream.close();
 *     }
 *   }, 1000);
 *
 *   stream.onAbort(() => {
 *     clearInterval(interval);
 *   });
 * }));
 * ```
 *
 * @example SSE with typed output schema
 * ```typescript
 * import { createRouter, sse } from '@agentuity/runtime';
 * import { s } from '@agentuity/schema';
 *
 * // Define your SSE event schema
 * export const outputSchema = s.object({
 *   type: s.enum(['token', 'complete', 'error']),
 *   content: s.optional(s.string()),
 * });
 *
 * const router = createRouter();
 *
 * // Pass schema as first argument for typed SSE routes
 * router.get('/stream', sse({ output: outputSchema }, async (c, stream) => {
 *   await stream.writeSSE({ data: JSON.stringify({ type: 'token', content: 'Hello' }) });
 *   await stream.writeSSE({ data: JSON.stringify({ type: 'complete' }) });
 *   stream.close();
 * }));
 *
 * // On the frontend, useEventStream will now have typed data:
 * // const { data } = useEventStream('/api/stream');
 * // data.type is 'token' | 'complete' | 'error'
 * ```
 *
 * @param handler - Handler function receiving context and SSE stream
 * @param options - Optional configuration with output schema for type inference
 * @returns Hono handler for SSE streaming
 * @see https://github.com/agentuity/sdk/issues/471
 * @see https://github.com/agentuity/sdk/issues/855
 */
export declare function sse<E extends Env = Env>(handler: SSEHandler<E>): Handler<E>;
/**
 * Creates an SSE middleware with typed output schema.
 *
 * @param options - Configuration object containing the output schema
 * @param handler - Handler function receiving context and SSE stream
 * @returns Hono handler for SSE streaming
 */
export declare function sse<E extends Env = Env, TOutput = unknown>(options: SSEOptions<TOutput>, handler: SSEHandler<E>): Handler<E>;
//# sourceMappingURL=sse.d.ts.map