import { z } from 'zod';
import { type APIClient } from '../api';
import { type CreateSourceRequest, type QueueApiOptions, type Source, type UpdateSourceRequest } from './types';
export declare const SourceResponseSchema: z.ZodDiscriminatedUnion<[z.ZodObject<{
    success: z.ZodLiteral<false>;
    message: z.ZodString;
    code: z.ZodOptional<z.ZodString>;
}, z.core.$strip>, z.ZodObject<{
    success: z.ZodLiteral<true>;
    data: z.ZodObject<{
        source: z.ZodObject<{
            id: z.ZodString;
            queue_id: z.ZodString;
            name: z.ZodString;
            description: z.ZodOptional<z.ZodNullable<z.ZodString>>;
            auth_type: z.ZodEnum<{
                none: "none";
                basic: "basic";
                header: "header";
            }>;
            enabled: z.ZodBoolean;
            url: z.ZodString;
            request_count: z.ZodNumber;
            success_count: z.ZodNumber;
            failure_count: z.ZodNumber;
            last_request_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
            last_success_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
            last_failure_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
            last_failure_error: z.ZodOptional<z.ZodNullable<z.ZodString>>;
            created_at: z.ZodString;
            updated_at: z.ZodString;
        }, z.core.$strip>;
    }, z.core.$strip>;
}, z.core.$strip>], "success">;
export declare const SourcesListResponseSchema: z.ZodDiscriminatedUnion<[z.ZodObject<{
    success: z.ZodLiteral<false>;
    message: z.ZodString;
    code: z.ZodOptional<z.ZodString>;
}, z.core.$strip>, z.ZodObject<{
    success: z.ZodLiteral<true>;
    data: z.ZodObject<{
        sources: z.ZodArray<z.ZodObject<{
            id: z.ZodString;
            queue_id: z.ZodString;
            name: z.ZodString;
            description: z.ZodOptional<z.ZodNullable<z.ZodString>>;
            auth_type: z.ZodEnum<{
                none: "none";
                basic: "basic";
                header: "header";
            }>;
            enabled: z.ZodBoolean;
            url: z.ZodString;
            request_count: z.ZodNumber;
            success_count: z.ZodNumber;
            failure_count: z.ZodNumber;
            last_request_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
            last_success_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
            last_failure_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
            last_failure_error: z.ZodOptional<z.ZodNullable<z.ZodString>>;
            created_at: z.ZodString;
            updated_at: z.ZodString;
        }, z.core.$strip>>;
    }, z.core.$strip>;
}, z.core.$strip>], "success">;
export declare const DeleteSourceResponseSchema: z.ZodDiscriminatedUnion<[z.ZodObject<{
    success: z.ZodLiteral<false>;
    message: z.ZodString;
    code: z.ZodOptional<z.ZodString>;
}, z.core.$strip>, z.ZodObject<{
    success: z.ZodLiteral<true>;
}, z.core.$strip>], "success">;
/**
 * Create a source for a queue.
 *
 * Sources are public HTTP ingestion endpoints that allow external systems to
 * publish messages to a queue. They support various authentication methods
 * to secure access.
 *
 * @param client - The API client instance
 * @param queueName - The name of the queue to add the source to
 * @param params - Source configuration including name and optional auth settings
 * @returns The created source with assigned ID and public URL
 * @throws {QueueValidationError} If validation fails (invalid queue name or source name)
 * @throws {QueueNotFoundError} If the queue does not exist
 * @throws {SourceAlreadyExistsError} If a source with the same name already exists
 * @throws {QueueError} If the API request fails
 *
 * @example
 * ```typescript
 * const source = await createSource(client, 'order-events', {
 *   name: 'webhook-ingestion',
 *   description: 'Receives webhooks from external service',
 *   auth_type: 'header',
 *   auth_value: 'Bearer my-secret-token',
 * });
 * console.log(`Created source ${source.id} at ${source.url}`);
 * ```
 */
export declare function createSource(client: APIClient, queueName: string, params: CreateSourceRequest, options?: QueueApiOptions): Promise<Source>;
/**
 * List all sources for a queue.
 *
 * Retrieves all HTTP ingestion endpoints configured for a queue. Each source
 * provides a public URL for external systems to publish messages.
 *
 * @param client - The API client instance
 * @param queueName - The name of the queue
 * @returns Array of sources configured for the queue
 * @throws {QueueValidationError} If validation fails (invalid queue name)
 * @throws {QueueNotFoundError} If the queue does not exist
 * @throws {QueueError} If the API request fails
 *
 * @example
 * ```typescript
 * const sources = await listSources(client, 'order-events');
 * for (const source of sources) {
 *   console.log(`Source ${source.id}: ${source.name} (${source.enabled ? 'enabled' : 'disabled'})`);
 *   console.log(`  URL: ${source.url}`);
 *   console.log(`  Success rate: ${source.success_count}/${source.request_count}`);
 * }
 * ```
 */
export declare function listSources(client: APIClient, queueName: string, options?: QueueApiOptions): Promise<Source[]>;
/**
 * Get a source by ID.
 *
 * Retrieves a specific source's details including its public URL and statistics.
 *
 * @param client - The API client instance
 * @param queueName - The name of the queue
 * @param sourceId - The source ID to retrieve (prefixed with qsrc_)
 * @returns The source details
 * @throws {QueueValidationError} If validation fails (invalid queue name or source ID)
 * @throws {SourceNotFoundError} If the source does not exist
 * @throws {QueueNotFoundError} If the queue does not exist
 * @throws {QueueError} If the API request fails
 *
 * @example
 * ```typescript
 * const source = await getSource(client, 'order-events', 'qsrc_abc123');
 * console.log(`Source: ${source.name}`);
 * console.log(`URL: ${source.url}`);
 * console.log(`Auth type: ${source.auth_type}`);
 * console.log(`Last request: ${source.last_request_at}`);
 * ```
 */
export declare function getSource(client: APIClient, queueName: string, sourceId: string, options?: QueueApiOptions): Promise<Source>;
/**
 * Update a source's configuration.
 *
 * Modifies an existing source's settings such as name, enabled status,
 * or authentication configuration. Only the fields provided in params will be updated.
 *
 * @param client - The API client instance
 * @param queueName - The name of the queue
 * @param sourceId - The source ID to update (prefixed with qsrc_)
 * @param params - Fields to update (partial update supported)
 * @returns The updated source
 * @throws {QueueValidationError} If validation fails (invalid queue name or source ID)
 * @throws {SourceNotFoundError} If the source does not exist
 * @throws {QueueNotFoundError} If the queue does not exist
 * @throws {QueueError} If the API request fails
 *
 * @example
 * ```typescript
 * // Disable a source temporarily
 * const updated = await updateSource(client, 'order-events', 'qsrc_abc123', {
 *   enabled: false,
 * });
 *
 * // Update authentication
 * const updated = await updateSource(client, 'order-events', 'qsrc_abc123', {
 *   auth_type: 'basic',
 *   auth_value: 'user:password',
 * });
 * ```
 */
export declare function updateSource(client: APIClient, queueName: string, sourceId: string, params: UpdateSourceRequest, options?: QueueApiOptions): Promise<Source>;
/**
 * Delete a source from a queue.
 *
 * Permanently removes an HTTP ingestion endpoint. The public URL will no longer
 * accept requests. This action cannot be undone.
 *
 * @param client - The API client instance
 * @param queueName - The name of the queue
 * @param sourceId - The source ID to delete (prefixed with qsrc_)
 * @returns void
 * @throws {QueueValidationError} If validation fails (invalid queue name or source ID)
 * @throws {SourceNotFoundError} If the source does not exist
 * @throws {QueueNotFoundError} If the queue does not exist
 * @throws {QueueError} If the API request fails
 *
 * @example
 * ```typescript
 * await deleteSource(client, 'order-events', 'qsrc_abc123');
 * console.log('Source deleted');
 * ```
 */
export declare function deleteSource(client: APIClient, queueName: string, sourceId: string, options?: QueueApiOptions): Promise<void>;
//# sourceMappingURL=sources.d.ts.map