import { type OrganizationList } from '@agentuity/server';
import type { ColorScheme } from './terminal';
import type { Profile } from './types';
import { type APIClient as APIClientType } from './api';
import { maskSecret } from './env-util';
export { maskSecret };
export { createPrompt, PromptFlow } from './tui/prompt';
export { group } from './tui/group';
export { note, drawBox, errorBox, warningBox } from './tui/box';
export { symbols } from './tui/symbols';
export { colors as tuiColors } from './tui/colors';
export type { TextOptions, ConfirmOptions, SelectOptions, SelectOption, MultiSelectOptions, } from './tui/prompt';
export declare const ICONS: {
    readonly success: "[OK]";
    readonly error: "[ERROR]";
    readonly warning: "[WARN]";
    readonly info: "[INFO]";
    readonly arrow: "->";
    readonly bullet: "-";
} | {
    readonly success: "✓";
    readonly error: "✗";
    readonly warning: "⚠";
    readonly info: "ℹ";
    readonly arrow: "→";
    readonly bullet: "•";
};
/**
 * Check if we should treat the terminal as TTY-like for interactive output
 * (real TTY on stdout or stderr, or FORCE_COLOR set by fork wrapper).
 * Returns false in CI environments since CI terminals often don't support
 * cursor control sequences reliably.
 * Returns false when running from an AI coding agent (no interactive prompts/spinners).
 */
export declare function isTTYLike(): boolean;
/**
 * Get terminal width, respecting COLUMNS env var for piped processes
 */
export declare function getTerminalWidth(defaultWidth?: number): number;
export declare function shouldUseColors(): boolean;
declare function getColors(): {
    readonly success: {
        readonly light: "";
        readonly dark: "";
    };
    readonly error: {
        readonly light: "";
        readonly dark: "";
    };
    readonly warning: {
        readonly light: "";
        readonly dark: "";
    };
    readonly info: {
        readonly light: "";
        readonly dark: "";
    };
    readonly muted: {
        readonly light: "";
        readonly dark: "";
    };
    readonly bold: {
        readonly light: "";
        readonly dark: "";
    };
    readonly link: {
        readonly light: "";
        readonly dark: "";
    };
    readonly primary: {
        readonly light: "";
        readonly dark: "";
    };
    readonly reset: "";
} | {
    readonly success: {
        readonly light: "\u001B[32m";
        readonly dark: "\u001B[92m";
    };
    readonly error: {
        readonly light: "\u001B[31m";
        readonly dark: "\u001B[91m";
    };
    readonly warning: {
        readonly light: "\u001B[33m";
        readonly dark: "\u001B[93m";
    };
    readonly info: {
        readonly light: "\u001B[36m";
        readonly dark: "\u001B[96m";
    };
    readonly muted: {
        readonly light: "\u001B[90m";
        readonly dark: "\u001B[90m";
    };
    readonly bold: {
        readonly light: "\u001B[1m";
        readonly dark: "\u001B[1m";
    };
    readonly link: {
        readonly light: "\u001B[34;4m";
        readonly dark: "\u001B[94;4m";
    };
    readonly primary: {
        readonly light: "\u001B[30m";
        readonly dark: "\u001B[97m";
    };
    readonly reset: "\u001B[0m";
};
export declare function setColorScheme(scheme: ColorScheme): void;
export declare function isDarkMode(): boolean;
export declare function getColor(colorKey: keyof ReturnType<typeof getColors>): string;
/**
 * Color helpers that return colored strings (for inline use, no icons)
 */
export declare function colorSuccess(text: string): string;
export declare function colorError(text: string): string;
export declare function colorWarning(text: string): string;
export declare function colorInfo(text: string): string;
export declare function colorMuted(text: string): string;
export declare function colorPrimary(text: string): string;
/**
 * Get the appropriate color function for a log severity level
 */
export declare function getSeverityColor(severity: string): (text: string) => string;
/**
 * Print a success message with a green checkmark
 */
export declare function success(message: string): void;
/**
 * Print an error message with a red X
 */
export declare function error(message: string): void;
/**
 * Print an error message with a red X and then exit
 */
export declare function fatal(message: string, errorCode?: import('./errors').ErrorCode): never;
/**
 * Print a warning message with a yellow warning icon
 */
export declare function warning(message: string, asError?: boolean): void;
/**
 * Print an info message with a cyan info icon
 */
export declare function info(message: string): void;
/**
 * Format text in muted/gray color
 */
export declare function muted(text: string): string;
/**
 * Format text in warn color
 */
export declare function warn(text: string): string;
/**
 * Format text in bold
 */
export declare function bold(text: string): string;
/**
 * Format text with white bold (or inverse for light mode)
 * Used for table headings
 */
export declare function heading(text: string): string;
/**
 * Format text as a link (blue and underlined)
 */
export declare function link(url: string, title?: string, color?: string): string;
/**
 * Check if terminal supports OSC 8 hyperlinks
 */
export declare function supportsHyperlinks(): boolean;
export declare function fileUrl(file: string, line?: number, col?: number): string;
export declare function sourceLink(file: string, line: number, col: number, display?: string, color?: string): string;
/**
 * Print a bulleted list item
 */
export declare function bullet(message: string): void;
/**
 * Print an arrow item (for showing next steps)
 */
export declare function arrow(message: string): void;
/**
 * Print a blank line
 */
export declare function newline(): void;
/**
 * Print plain text output without any prefix or icon
 * Use for primary command output that shouldn't have semantic formatting
 */
export declare function output(message: string): void;
/**
 * Get the display width of a string, handling ANSI codes and OSC 8 hyperlinks
 *
 * Note: Bun.stringWidth() counts OSC 8 hyperlink escape sequences in the width,
 * which causes incorrect alignment. We strip OSC 8 codes first, then use Bun.stringWidth()
 * to handle regular ANSI codes and unicode characters correctly.
 */
export declare function getDisplayWidth(str: string): number;
/**
 * Strip all ANSI escape sequences from a string
 */
export declare function stripAnsi(str: string): string;
/**
 * Truncate a string to a maximum display width, handling ANSI codes and Unicode correctly
 * Preserves ANSI escape sequences and doesn't break multi-byte characters or grapheme clusters
 */
export declare function truncateToWidth(str: string, maxWidth: number, ellipsis?: string): string;
/**
 * Pad a string to a specific length on the right
 */
export declare function padRight(str: string, length: number, pad?: string): string;
/**
 * Pad a string to a specific length on the left
 */
export declare function padLeft(str: string, length: number, pad?: string): string;
interface BannerOptions {
    padding?: number;
    minWidth?: number;
    topSpacer?: boolean;
    middleSpacer?: boolean;
    bottomSpacer?: boolean;
    centerTitle?: boolean;
}
/**
 * Display a formatted banner with title and body content
 * Creates a bordered box around the content
 *
 * Uses Bun.stringWidth() for accurate width calculation with ANSI codes and unicode
 * Responsive to terminal width - adapts to narrow terminals
 */
export declare function banner(title: string, body: string, options?: BannerOptions): void;
/**
 * Wait for any key press before continuing
 * Displays a prompt message and waits for user input
 * Exits with code 1 if CTRL+C is pressed
 */
export declare function waitForAnyKey(message?: string): Promise<void>;
/**
 * Prompts user with a yes/no question
 * Returns true for yes, false for no
 * Exits with code 1 if CTRL+C is pressed
 */
export declare function confirm(message: string, defaultValue?: boolean): Promise<boolean>;
/**
 * Display a signup benefits box with cyan border
 * Shows the value proposition for creating an Agentuity account
 */
export declare function showSignupBenefits(): void;
/**
 * Display a message when unauthenticated to let the user know certain capabilities are disabled
 * @param hasProfile - If true, user has logged in before so only show "Login" instead of "Sign up / Login"
 */
export declare function showLoggedOutMessage(appBaseUrl: string, hasProfile?: boolean): void;
/**
 * Display a warning when running in local-only mode (no agentuity.json project config)
 * This is shown during `agentuity dev` when the project hasn't been registered with Agentuity Cloud
 */
export declare function showLocalOnlyWarning(): void;
/**
 * Copy text to clipboard
 * Returns true if successful, false otherwise
 */
export declare function copyToClipboard(text: string): Promise<boolean>;
/**
 * Wrap text to a maximum width
 * Handles explicit newlines and word wrapping
 * Preserves ANSI color codes across wrapped lines
 */
export declare function wrapText(text: string, maxWidth: number): string[];
/**
 * Progress callback for spinner
 */
export type SpinnerProgressCallback = (progress: number) => void;
/**
 * Log callback for spinner
 */
export type SpinnerLogCallback = (message: string) => void;
/**
 * Spinner options (simple without progress)
 */
export interface SimpleSpinnerOptions<T> {
    type?: 'simple';
    message: string;
    callback: (() => Promise<T>) | Promise<T>;
    /**
     * If true, clear the spinner output on success (no icon, no message)
     * Defaults to false
     */
    clearOnSuccess?: boolean;
    /**
     * If true, suppress the error message display on failure (for custom error handling)
     * Defaults to false
     */
    clearOnError?: boolean;
}
/**
 * Spinner options (with progress tracking)
 */
export interface ProgressSpinnerOptions<T> {
    type: 'progress';
    message: string;
    callback: (progress: SpinnerProgressCallback) => Promise<T>;
    /**
     * If true, clear the spinner output on success (no icon, no message)
     * Defaults to false
     */
    clearOnSuccess?: boolean;
    /**
     * If true, suppress the error message display on failure (for custom error handling)
     * Defaults to false
     */
    clearOnError?: boolean;
}
/**
 * Spinner options (with logger streaming)
 */
export interface LoggerSpinnerOptions<T> {
    type: 'logger';
    message: string;
    callback: (log: SpinnerLogCallback) => Promise<T>;
    /**
     * If true, clear the spinner output on success (no icon, no message)
     * Defaults to false
     */
    clearOnSuccess?: boolean;
    /**
     * Maximum number of log lines to show while running
     * If < 0, shows all lines. Defaults to 3.
     */
    maxLines?: number;
}
/**
 * Spinner options (with countdown timer)
 */
export interface CountdownSpinnerOptions<T> {
    type: 'countdown';
    message: string;
    timeoutMs: number;
    callback: () => Promise<T>;
    /**
     * If true, clear the spinner output on success (no icon, no message)
     * Defaults to false
     */
    clearOnSuccess?: boolean;
    /**
     * Optional callback to handle Enter key press
     * Can be used to open a URL in the browser
     */
    onEnterPress?: () => void;
}
/**
 * Spinner options (discriminated union)
 */
export type SpinnerOptions<T> = SimpleSpinnerOptions<T> | ProgressSpinnerOptions<T> | LoggerSpinnerOptions<T> | CountdownSpinnerOptions<T>;
/**
 * Run a callback with an animated spinner (simple overload)
 *
 * Shows a spinner animation while the callback executes.
 * On success, shows a checkmark. On error, shows an X and re-throws.
 *
 * @param message - The message to display next to the spinner
 * @param callback - Async function or Promise to execute
 */
export declare function spinner<T>(message: string, callback: (() => Promise<T>) | Promise<T>): Promise<T>;
/**
 * Run a callback with an animated spinner (options overload)
 *
 * Shows a spinner animation while the callback executes.
 * On success, shows a checkmark. On error, shows an X and re-throws.
 *
 * @param options - Spinner options with optional progress tracking
 */
export declare function spinner<T>(options: SpinnerOptions<T>): Promise<T>;
/**
 * Alias for spinner function (for better semantics when using progress/logger types)
 */
export declare const progress: typeof spinner;
/**
 * Options for running a command with streaming output
 */
export interface CommandRunnerOptions {
    /**
     * The command to run (displayed in the UI)
     */
    command: string;
    /**
     * The actual command and arguments to execute
     */
    cmd: string[];
    /**
     * Current working directory
     */
    cwd?: string;
    /**
     * Environment variables
     */
    env?: Record<string, string>;
    /**
     * If true, clear output on success and only show command + success icon
     * Defaults to false
     */
    clearOnSuccess?: boolean;
    /**
     * If true or undefined, will truncate each line of output
     */
    truncate?: boolean;
    /**
     * If undefined, will show up to 3 last lines of output while running. Customize the number with this property.
     */
    maxLinesOutput?: number;
    /**
     * If undefined, will show up to 10 last lines on failure. Customize the number with this property.
     */
    maxLinesOnFailure?: number;
}
/**
 * Run an external command and stream its output with a live UI
 *
 * Displays the command with a colored $ prompt:
 * - Blue while running
 * - Green on successful exit (code 0)
 * - Red on failed exit (code != 0)
 *
 * Shows the last 3 lines of output as it streams.
 */
export declare function runCommand(options: CommandRunnerOptions): Promise<number>;
/**
 * Prompt user for text input
 * Returns the input string
 */
export declare function prompt(message: string): Promise<string>;
/**
 * Select an organization from a list.
 *
 * @param orgs - List of organizations to choose from
 * @param initial - Preferred org ID to pre-select (from saved preferences)
 * @param autoSelect - If true, auto-select preferred org without prompting (for --confirm or non-interactive)
 * @returns The selected organization ID
 */
export declare function selectOrganization(orgs: OrganizationList, initial?: string, autoSelect?: boolean): Promise<string>;
/**
 * show a project list picker
 *
 * @param apiClient
 * @param showDeployment
 * @returns
 */
export declare function showProjectList(apiClient: APIClientType, showDeploymentId?: boolean): Promise<string>;
/**
 * Show a profile list picker
 *
 * @param profiles List of profiles to choose from
 * @param message Prompt message
 * @returns The name of the selected profile
 */
export declare function showProfileList(profiles: Profile[], message?: string): Promise<string>;
export declare function json(value: unknown): void;
export declare function plural(count: number, singular: string, plural: string): string;
/**
 * Table column definition
 */
export interface TableColumn {
    /** Column name */
    name: string;
    /** Column alignment */
    alignment?: 'left' | 'right' | 'center';
}
/**
 * Table options
 */
export interface TableOptions {
    /**
     * If true, returns the table as a string instead of printing to stdout
     */
    render?: boolean;
    /**
     * Force a specific layout mode
     * - 'horizontal': Traditional table with columns side by side
     * - 'vertical': Stacked format with "Column: value" on separate lines
     * - 'auto': Automatically choose based on terminal width (default)
     */
    layout?: 'horizontal' | 'vertical' | 'auto';
    /**
     * the padding before any label
     */
    padStart?: string;
}
/**
 * Display data in a formatted table using cli-table3
 *
 * Supports two modes:
 * 1. Simple mode: Pass data array and optional column names
 * 2. Advanced mode: Pass column configurations with custom names and alignment
 *
 * Automatically switches between horizontal (wide) and vertical (narrow) layouts
 * based on terminal width. Use the `layout` option to force a specific mode.
 *
 * @param data - Array of data objects to display
 * @param columns - Column names or column configurations
 * @param options - Additional options
 * @returns If render=true, returns the table as a string, otherwise prints to stdout
 */
export declare function table<T extends Record<string, unknown>>(data: T[], columns?: (keyof T)[] | TableColumn[], options?: TableOptions): string | void;
export declare function formatBytes(bytes: number): string;
export declare function clearLastLines(n: number, s?: (v: string) => void): void;
//# sourceMappingURL=tui.d.ts.map