/**
 * Build Report
 *
 * Structured error/warning collection and reporting for build and deploy commands.
 * Outputs a JSON file with a strict schema for CI tooling integration.
 */
import type { GrammarItem } from './tsc-output-parser';
/**
 * Error codes for non-TypeScript errors.
 * TypeScript errors use their native TS#### codes.
 */
export declare const BuildErrorCodes: {
    readonly AST001: "MetadataNameMissing";
    readonly AST002: "DuplicateName";
    readonly AST003: "InvalidExport";
    readonly AST004: "InvalidCronExpression";
    readonly AST005: "InvalidAgentConfig";
    readonly BUILD001: "AppFileNotFound";
    readonly BUILD002: "SourceDirNotFound";
    readonly BUILD003: "DependencyUpgradeFailed";
    readonly BUILD004: "BundleFailed";
    readonly BUILD005: "EntryPointNotFound";
    readonly BUILD006: "ViteBuildFailed";
    readonly BUILD007: "RuntimePackageNotFound";
    readonly BUILD008: "TypecheckToolFailed";
    readonly VAL001: "AgentIdentifierCollision";
    readonly VAL002: "InvalidRoutePath";
    readonly VAL003: "InvalidRouteMethod";
    readonly VAL004: "SchemaValidationFailed";
    readonly DEPLOY001: "DeploymentCreationFailed";
    readonly DEPLOY002: "UploadFailed";
    readonly DEPLOY003: "DeploymentTimeout";
    readonly DEPLOY004: "DeploymentFailed";
    readonly DEPLOY005: "EncryptionFailed";
    readonly DEPLOY006: "CDNUploadFailed";
};
export type BuildErrorCode = keyof typeof BuildErrorCodes;
/**
 * Error scopes for categorizing errors
 */
export type ErrorScope = 'typescript' | 'ast' | 'build' | 'bundler' | 'validation' | 'deploy';
/**
 * File-specific error with location information
 */
export interface FileError {
    type: 'file';
    scope: ErrorScope;
    path: string;
    line: number;
    column: number;
    message: string;
    code?: string;
}
/**
 * General error without file location
 */
export interface GeneralError {
    type: 'general';
    scope: ErrorScope;
    message: string;
    code?: string;
}
/**
 * Union type for all build errors
 */
export type BuildError = FileError | GeneralError;
/**
 * Union type for all build warnings (same structure as errors)
 */
export type BuildWarning = FileError | GeneralError;
/**
 * Diagnostic timing information for a build phase
 */
export interface BuildDiagnostic {
    name: string;
    startedAt: string;
    completedAt: string;
    durationMs: number;
}
/**
 * Complete build report structure
 */
export interface BuildReport {
    success: boolean;
    errors: BuildError[];
    warnings: BuildWarning[];
    diagnostics: BuildDiagnostic[];
}
/**
 * Diagnostic phases for timing
 */
export declare const DiagnosticPhases: readonly ["typecheck", "client-build", "workbench-build", "server-build", "metadata-generation", "zip-package", "encrypt", "code-upload", "cdn-upload", "deployment-wait"];
export type DiagnosticPhase = (typeof DiagnosticPhases)[number];
/**
 * Build Report Collector
 *
 * Collects errors, warnings, and diagnostic timing information throughout
 * the build/deploy pipeline. Can be configured to automatically write
 * the report on process exit.
 */
export declare class BuildReportCollector {
    private errors;
    private warnings;
    private diagnostics;
    private activeDiagnostics;
    private outputPath;
    private autoWriteEnabled;
    private written;
    private beforeExitHandler;
    private sigintHandler;
    private sigtermHandler;
    /**
     * Set the output path for the report file
     */
    setOutputPath(path: string): void;
    /**
     * Enable automatic writing of the report on process exit.
     * This ensures the report is written even if the process exits unexpectedly.
     */
    enableAutoWrite(): void;
    /**
     * Disable automatic writing and remove signal handlers.
     * Call this when done with the collector to prevent handler conflicts.
     */
    disableAutoWrite(): void;
    /**
     * Add TypeScript errors from parsed tsc output
     */
    addTypeScriptErrors(items: GrammarItem[]): void;
    /**
     * Add a file-specific error
     */
    addFileError(scope: ErrorScope, path: string, line: number, column: number, message: string, code?: string): void;
    /**
     * Add a general error without file location
     */
    addGeneralError(scope: ErrorScope, message: string, code?: string): void;
    /**
     * Add a file-specific warning
     */
    addFileWarning(scope: ErrorScope, path: string, line: number, column: number, message: string, code?: string): void;
    /**
     * Add a general warning without file location
     */
    addGeneralWarning(scope: ErrorScope, message: string, code?: string): void;
    /**
     * Start timing a diagnostic phase
     * @returns A function to call when the phase completes
     */
    startDiagnostic(name: string): () => void;
    /**
     * End a diagnostic phase
     */
    private endDiagnostic;
    /**
     * Check if there are any errors
     */
    hasErrors(): boolean;
    /**
     * Check if there are any warnings
     */
    hasWarnings(): boolean;
    /**
     * Get the error count
     */
    getErrorCount(): number;
    /**
     * Get the warning count
     */
    getWarningCount(): number;
    /**
     * Generate the complete build report
     */
    toReport(): BuildReport;
    /**
     * Write the report to the configured output path asynchronously
     */
    write(): Promise<void>;
    /**
     * Write the report synchronously (for exit handlers)
     */
    writeSync(): void;
    /**
     * Force write the report (bypasses the written flag)
     * Use this when you want to update the report file mid-process
     */
    forceWrite(): Promise<void>;
}
/**
 * Set the global collector instance
 */
export declare function setGlobalCollector(collector: BuildReportCollector): void;
/**
 * Get the global collector instance (may be null)
 */
export declare function getGlobalCollector(): BuildReportCollector | null;
/**
 * Clear the global collector instance and clean up its signal handlers
 */
export declare function clearGlobalCollector(): void;
//# sourceMappingURL=build-report.d.ts.map