/**
 * General webhook operation error.
 *
 * Thrown when a webhook operation fails for reasons other than not-found.
 *
 * @example
 * ```typescript
 * try {
 *   await createWebhook(client, { name: 'my-webhook' });
 * } catch (error) {
 *   if (error instanceof WebhookError) {
 *     console.error(`Webhook operation failed: ${error.message}`);
 *   }
 * }
 * ```
 */
export declare const WebhookError: {
    new (args?: ({
        webhookId?: string;
    } & {
        message?: string;
        cause?: unknown;
    }) | undefined): import("@agentuity/core").RichError & {
        readonly _tag: "WebhookError";
    } & Readonly<{
        webhookId?: string;
    }>;
    readonly defaultMessage?: string;
};
/**
 * Error thrown when a webhook is not found.
 *
 * @example
 * ```typescript
 * try {
 *   await getWebhook(client, 'wh_nonexistent');
 * } catch (error) {
 *   if (error instanceof WebhookNotFoundError) {
 *     console.error(`Webhook not found: ${error.webhookId}`);
 *   }
 * }
 * ```
 */
export declare const WebhookNotFoundError: {
    new (args?: ({
        webhookId: string;
    } & {
        message?: string;
        cause?: unknown;
    }) | undefined): import("@agentuity/core").RichError & {
        readonly _tag: "WebhookNotFoundError";
    } & Readonly<{
        webhookId: string;
    }>;
    readonly defaultMessage?: string;
};
/**
 * Error thrown when a webhook destination is not found.
 *
 * @example
 * ```typescript
 * try {
 *   await deleteWebhookDestination(client, 'wh_abc', 'whds_nonexistent');
 * } catch (error) {
 *   if (error instanceof WebhookDestinationNotFoundError) {
 *     console.error(`Destination ${error.destinationId} not found`);
 *   }
 * }
 * ```
 */
export declare const WebhookDestinationNotFoundError: {
    new (args?: ({
        webhookId: string;
        destinationId: string;
    } & {
        message?: string;
        cause?: unknown;
    }) | undefined): import("@agentuity/core").RichError & {
        readonly _tag: "WebhookDestinationNotFoundError";
    } & Readonly<{
        webhookId: string;
        destinationId: string;
    }>;
    readonly defaultMessage?: string;
};
/**
 * Error thrown when a webhook receipt is not found.
 *
 * @example
 * ```typescript
 * try {
 *   await getWebhookReceipt(client, 'wh_abc', 'whrc_nonexistent');
 * } catch (error) {
 *   if (error instanceof WebhookReceiptNotFoundError) {
 *     console.error(`Receipt ${error.receiptId} not found`);
 *   }
 * }
 * ```
 */
export declare const WebhookReceiptNotFoundError: {
    new (args?: ({
        webhookId: string;
        receiptId: string;
    } & {
        message?: string;
        cause?: unknown;
    }) | undefined): import("@agentuity/core").RichError & {
        readonly _tag: "WebhookReceiptNotFoundError";
    } & Readonly<{
        webhookId: string;
        receiptId: string;
    }>;
    readonly defaultMessage?: string;
};
/**
 * Error thrown when a webhook delivery is not found.
 *
 * @example
 * ```typescript
 * try {
 *   await retryWebhookDelivery(client, 'wh_abc', 'whdv_nonexistent');
 * } catch (error) {
 *   if (error instanceof WebhookDeliveryNotFoundError) {
 *     console.error(`Delivery ${error.deliveryId} not found`);
 *   }
 * }
 * ```
 */
export declare const WebhookDeliveryNotFoundError: {
    new (args?: ({
        webhookId: string;
        deliveryId: string;
    } & {
        message?: string;
        cause?: unknown;
    }) | undefined): import("@agentuity/core").RichError & {
        readonly _tag: "WebhookDeliveryNotFoundError";
    } & Readonly<{
        webhookId: string;
        deliveryId: string;
    }>;
    readonly defaultMessage?: string;
};
/**
 * Constructs a full API path for webhook operations.
 *
 * Pattern: `/webhook/${VERSION}/${verb}/${segments.join('/')}`
 * Each verb is unique to avoid route conflicts with the Go mux.
 *
 * @param verb - The action verb (e.g., 'create', 'list', 'get', 'destination-list')
 * @param segments - Additional path segments (e.g., webhook ID, sub-resource ID)
 * @returns The full API path with version and verb prefix
 *
 * @internal
 */
export declare function webhookApiPath(verb: string, ...segments: string[]): string;
/**
 * Constructs a full API path for webhook operations with query string.
 *
 * @param verb - The action verb
 * @param queryString - Query string to append (without leading ?)
 * @param segments - Additional path segments
 * @returns The full API path with version, verb, and query string
 *
 * @internal
 */
export declare function webhookApiPathWithQuery(verb: string, queryString: string | undefined, ...segments: string[]): string;
/**
 * Constructs the public ingest URL for a webhook.
 *
 * Pattern: `/webhook/{orgId}-{webhookId}` (non-versioned, public)
 *
 * @param orgId - The organization ID
 * @param webhookId - The webhook ID (prefixed with wh_)
 * @returns The public ingest URL
 *
 * @internal
 */
export declare function webhookIngestPath(orgId: string, webhookId: string): string;
/**
 * Builds headers for webhook API requests.
 *
 * @param orgId - Optional organization ID for CLI authentication
 * @returns Headers object to pass to API client
 *
 * @internal
 */
export declare function buildWebhookHeaders(orgId?: string): Record<string, string> | undefined;
/**
 * Wraps an API call and translates APIError with HTTP status codes to domain-specific webhook errors.
 *
 * - 404 → WebhookNotFoundError / WebhookDestinationNotFoundError / WebhookReceiptNotFoundError / WebhookDeliveryNotFoundError
 *
 * @internal
 */
export declare function withWebhookErrorHandling<T>(apiCall: () => Promise<T>, context: {
    webhookId?: string;
    destinationId?: string;
    receiptId?: string;
    deliveryId?: string;
}): Promise<T>;
//# sourceMappingURL=util.d.ts.map