import { z } from 'zod';
import { APIResponseSchema } from "../api.js";
import { API_VERSION, throwSandboxError } from "./util.js";
const SandboxStatusDataSchema = z.object({
    sandboxId: z.string(),
    status: z.string(),
    exitCode: z.number().optional(),
});
const SandboxStatusResponseSchema = APIResponseSchema(SandboxStatusDataSchema);
/**
 * Retrieves lightweight sandbox status (status + exitCode only).
 * Optimized for the sandbox run flow — backed by Redis for ~1ms response time.
 */
export async function sandboxGetStatus(client, params) {
    const { sandboxId, orgId } = params;
    const queryParams = new URLSearchParams();
    if (orgId) {
        queryParams.set('orgId', orgId);
    }
    const queryString = queryParams.toString();
    const url = `/sandbox/${API_VERSION}/status/${sandboxId}${queryString ? `?${queryString}` : ''}`;
    const resp = await client.get(url, SandboxStatusResponseSchema);
    if (resp.success) {
        return {
            sandboxId: resp.data.sandboxId,
            status: resp.data.status,
            exitCode: resp.data.exitCode,
        };
    }
    throwSandboxError(resp, { sandboxId });
}
//# sourceMappingURL=getStatus.js.map