import { z } from 'zod';
import { APIResponseSchema, APIResponseSchemaNoData } from '../api';
import { API_VERSION, throwSandboxError } from './util';
// --- Schemas ---
export const DiskCheckpointInfoSchema = z.object({
    id: z.string().describe('Globally unique checkpoint ID (ckpt_xxx)'),
    name: z.string().describe('User-provided checkpoint name'),
    createdAt: z.string().describe('ISO timestamp of creation'),
    parent: z.string().describe('Parent checkpoint name (empty for base)'),
});
const CreateDiskCheckpointDataSchema = DiskCheckpointInfoSchema;
export const CreateDiskCheckpointResponseSchema = APIResponseSchema(CreateDiskCheckpointDataSchema);
const ListDiskCheckpointsDataSchema = z.object({
    checkpoints: z.array(DiskCheckpointInfoSchema),
});
export const ListDiskCheckpointsResponseSchema = APIResponseSchema(ListDiskCheckpointsDataSchema);
export const RestoreDiskCheckpointResponseSchema = APIResponseSchemaNoData();
export const DeleteDiskCheckpointResponseSchema = APIResponseSchemaNoData();
// --- API Functions ---
/**
 * Creates a new disk checkpoint for a sandbox.
 *
 * @param client - The API client to use for the request
 * @param params - Parameters including the sandbox ID and checkpoint name
 * @throws {SandboxResponseError} If the sandbox is not found or checkpoint creation fails
 */
export async function diskCheckpointCreate(client, params) {
    const { sandboxId, name, orgId } = params;
    const queryParams = new URLSearchParams();
    if (orgId) {
        queryParams.set('orgId', orgId);
    }
    const queryString = queryParams.toString();
    const url = `/sandbox/${API_VERSION}/${encodeURIComponent(sandboxId)}/checkpoint${queryString ? `?${queryString}` : ''}`;
    const resp = await client.post(url, { name }, CreateDiskCheckpointResponseSchema);
    if (resp.success) {
        return resp.data;
    }
    throwSandboxError(resp, { sandboxId });
}
/**
 * Lists all disk checkpoints for a sandbox.
 *
 * @param client - The API client to use for the request
 * @param params - Parameters including the sandbox ID
 * @throws {SandboxResponseError} If the sandbox is not found or listing fails
 */
export async function diskCheckpointList(client, params) {
    const { sandboxId, orgId } = params;
    const queryParams = new URLSearchParams();
    if (orgId) {
        queryParams.set('orgId', orgId);
    }
    const queryString = queryParams.toString();
    const url = `/sandbox/${API_VERSION}/checkpoints/${encodeURIComponent(sandboxId)}${queryString ? `?${queryString}` : ''}`;
    const resp = await client.get(url, ListDiskCheckpointsResponseSchema);
    if (resp.success) {
        return resp.data.checkpoints;
    }
    throwSandboxError(resp, { sandboxId });
}
/**
 * Restores a sandbox to a specific disk checkpoint.
 *
 * @param client - The API client to use for the request
 * @param params - Parameters including the sandbox ID and checkpoint ID
 * @throws {SandboxResponseError} If the sandbox or checkpoint is not found, or restore fails
 */
export async function diskCheckpointRestore(client, params) {
    const { sandboxId, checkpointId, orgId } = params;
    const queryParams = new URLSearchParams();
    if (orgId) {
        queryParams.set('orgId', orgId);
    }
    const queryString = queryParams.toString();
    const url = `/sandbox/${API_VERSION}/${encodeURIComponent(sandboxId)}/checkpoint/${encodeURIComponent(checkpointId)}/restore${queryString ? `?${queryString}` : ''}`;
    const resp = await client.post(url, undefined, RestoreDiskCheckpointResponseSchema);
    if (resp.success) {
        return;
    }
    throwSandboxError(resp, { sandboxId });
}
/**
 * Deletes a disk checkpoint from a sandbox.
 *
 * @param client - The API client to use for the request
 * @param params - Parameters including the sandbox ID and checkpoint ID
 * @throws {SandboxResponseError} If the sandbox or checkpoint is not found, or deletion fails
 */
export async function diskCheckpointDelete(client, params) {
    const { sandboxId, checkpointId, orgId } = params;
    const queryParams = new URLSearchParams();
    if (orgId) {
        queryParams.set('orgId', orgId);
    }
    const queryString = queryParams.toString();
    const url = `/sandbox/${API_VERSION}/${encodeURIComponent(sandboxId)}/checkpoint/${encodeURIComponent(checkpointId)}${queryString ? `?${queryString}` : ''}`;
    const resp = await client.delete(url, DeleteDiskCheckpointResponseSchema);
    if (resp.success) {
        return;
    }
    throwSandboxError(resp, { sandboxId });
}
//# sourceMappingURL=disk-checkpoint.js.map