/**
 * General queue operation error.
 *
 * Thrown when a queue operation fails for reasons other than not-found or validation.
 *
 * @example
 * ```typescript
 * try {
 *   await createQueue(client, { queue_type: 'worker' });
 * } catch (error) {
 *   if (error instanceof QueueError) {
 *     console.error(`Queue operation failed: ${error.message}`);
 *   }
 * }
 * ```
 */
export declare const QueueError: {
    new (args?: ({
        queueName?: string;
    } & {
        message?: string;
        cause?: unknown;
    }) | undefined): import("@agentuity/core").RichError & {
        readonly _tag: "QueueError";
    } & Readonly<{
        queueName?: string;
    }>;
    readonly defaultMessage?: string;
};
/**
 * Error thrown when a queue is not found.
 *
 * @example
 * ```typescript
 * try {
 *   await getQueue(client, 'non-existent-queue');
 * } catch (error) {
 *   if (error instanceof QueueNotFoundError) {
 *     console.error(`Queue not found: ${error.queueName}`);
 *   }
 * }
 * ```
 */
export declare const QueueNotFoundError: {
    new (args?: ({
        queueName: string;
    } & {
        message?: string;
        cause?: unknown;
    }) | undefined): import("@agentuity/core").RichError & {
        readonly _tag: "QueueNotFoundError";
    } & Readonly<{
        queueName: string;
    }>;
    readonly defaultMessage?: string;
};
/**
 * Error thrown when a message is not found in a queue.
 *
 * @example
 * ```typescript
 * try {
 *   await getMessage(client, 'my-queue', 'msg_abc123');
 * } catch (error) {
 *   if (error instanceof MessageNotFoundError) {
 *     console.error(`Message ${error.messageId} not found in ${error.queueName}`);
 *   }
 * }
 * ```
 */
export declare const MessageNotFoundError: {
    new (args?: ({
        queueName: string;
        messageId: string;
    } & {
        message?: string;
        cause?: unknown;
    }) | undefined): import("@agentuity/core").RichError & {
        readonly _tag: "MessageNotFoundError";
    } & Readonly<{
        queueName: string;
        messageId: string;
    }>;
    readonly defaultMessage?: string;
};
/**
 * Error thrown when a destination is not found.
 *
 * @example
 * ```typescript
 * try {
 *   await deleteDestination(client, 'my-queue', 'dest_abc123');
 * } catch (error) {
 *   if (error instanceof DestinationNotFoundError) {
 *     console.error(`Destination ${error.destinationId} not found`);
 *   }
 * }
 * ```
 */
export declare const DestinationNotFoundError: {
    new (args?: ({
        queueName: string;
        destinationId: string;
    } & {
        message?: string;
        cause?: unknown;
    }) | undefined): import("@agentuity/core").RichError & {
        readonly _tag: "DestinationNotFoundError";
    } & Readonly<{
        queueName: string;
        destinationId: string;
    }>;
    readonly defaultMessage?: string;
};
/**
 * Error thrown when a destination with the same type and URL already exists.
 *
 * @example
 * ```typescript
 * try {
 *   await createDestination(client, 'my-queue', { url: 'https://example.com' });
 * } catch (error) {
 *   if (error instanceof DestinationAlreadyExistsError) {
 *     console.error(`Destination already exists: ${error.url}`);
 *   }
 * }
 * ```
 */
export declare const DestinationAlreadyExistsError: {
    new (args?: ({
        queueName: string;
        url?: string;
    } & {
        message?: string;
        cause?: unknown;
    }) | undefined): import("@agentuity/core").RichError & {
        readonly _tag: "DestinationAlreadyExistsError";
    } & Readonly<{
        queueName: string;
        url?: string;
    }>;
    readonly defaultMessage?: string;
};
/**
 * Error thrown when an invalid argument is provided to a queue operation.
 *
 * @example
 * ```typescript
 * try {
 *   await createQueue(client, { queue_type: 'invalid' as any });
 * } catch (error) {
 *   if (error instanceof QueueInvalidArgumentError) {
 *     console.error(`Invalid parameter: ${error.param}`);
 *   }
 * }
 * ```
 */
export declare const QueueInvalidArgumentError: {
    new (args?: ({
        queueName?: string;
        param?: string;
    } & {
        message?: string;
        cause?: unknown;
    }) | undefined): import("@agentuity/core").RichError & {
        readonly _tag: "QueueInvalidArgumentError";
    } & Readonly<{
        queueName?: string;
        param?: string;
    }>;
    readonly defaultMessage?: string;
};
/**
 * Error thrown when a source is not found.
 *
 * @example
 * ```typescript
 * try {
 *   await getSource(client, 'my-queue', 'qsrc_abc123');
 * } catch (error) {
 *   if (error instanceof SourceNotFoundError) {
 *     console.error(`Source ${error.sourceId} not found in ${error.queueName}`);
 *   }
 * }
 * ```
 */
export declare const SourceNotFoundError: {
    new (args?: ({
        queueName: string;
        sourceId: string;
    } & {
        message?: string;
        cause?: unknown;
    }) | undefined): import("@agentuity/core").RichError & {
        readonly _tag: "SourceNotFoundError";
    } & Readonly<{
        queueName: string;
        sourceId: string;
    }>;
    readonly defaultMessage?: string;
};
/**
 * Error thrown when a source with the same name already exists.
 *
 * @example
 * ```typescript
 * try {
 *   await createSource(client, 'my-queue', { name: 'existing-source' });
 * } catch (error) {
 *   if (error instanceof SourceAlreadyExistsError) {
 *     console.error(`Source "${error.name}" already exists in ${error.queueName}`);
 *   }
 * }
 * ```
 */
export declare const SourceAlreadyExistsError: {
    new (args?: ({
        queueName: string;
        name: string;
    } & {
        message?: string;
        cause?: unknown;
    }) | undefined): import("@agentuity/core").RichError & {
        readonly _tag: "SourceAlreadyExistsError";
    } & Readonly<{
        queueName: string;
        name: string;
    }>;
    readonly defaultMessage?: string;
};
/**
 * Constructs a full API path for queue operations.
 * New pattern: /queue/[action]/$VERSION/[args]
 *
 * @param action - The action/resource (e.g., 'create', 'list', 'get', 'messages/publish')
 * @param args - Optional path arguments (e.g., queue name, message id)
 * @returns The full API path with version prefix
 *
 * @internal
 */
export declare function queueApiPath(action: string, ...args: string[]): string;
/**
 * Constructs a full API path for queue operations with query string.
 *
 * @param action - The action/resource
 * @param queryString - Query string to append (without leading ?)
 * @param args - Optional path arguments
 * @returns The full API path with version prefix and query string
 *
 * @internal
 */
export declare function queueApiPathWithQuery(action: string, queryString: string | undefined, ...args: string[]): string;
/**
 * Builds headers for queue API requests.
 *
 * @param orgId - Optional organization ID for CLI authentication
 * @returns Headers object to pass to API client
 *
 * @internal
 */
export declare function buildQueueHeaders(orgId?: string): Record<string, string> | undefined;
/**
 * Wraps an API call and translates APIError with HTTP status codes to domain-specific queue errors.
 *
 * - 404 → QueueNotFoundError / MessageNotFoundError / DestinationNotFoundError / SourceNotFoundError
 * - 409 with "already exists" → DestinationAlreadyExistsError / SourceAlreadyExistsError
 *
 * @internal
 */
export declare function withQueueErrorHandling<T>(apiCall: () => Promise<T>, context: {
    queueName?: string;
    messageId?: string;
    destinationId?: string;
    sourceId?: string;
    sourceName?: string;
    destinationUrl?: string;
}): Promise<T>;
//# sourceMappingURL=util.d.ts.map