import { z } from 'zod';
import { type APIClient } from '../api.ts';
import { type CreateQueueRequest, type ListQueuesRequest, type Queue, type QueueApiOptions, type UpdateQueueRequest } from './types.ts';
export declare const QueueResponseSchema: z.ZodDiscriminatedUnion<[z.ZodObject<{
    success: z.ZodLiteral<false>;
    message: z.ZodString;
    code: z.ZodOptional<z.ZodString>;
}, z.core.$strip>, z.ZodObject<{
    success: z.ZodLiteral<true>;
    data: z.ZodObject<{
        queue: z.ZodObject<{
            id: z.ZodString;
            name: z.ZodString;
            description: z.ZodOptional<z.ZodNullable<z.ZodString>>;
            queue_type: z.ZodEnum<{
                worker: "worker";
                pubsub: "pubsub";
            }>;
            default_ttl_seconds: z.ZodOptional<z.ZodNullable<z.ZodNumber>>;
            default_visibility_timeout_seconds: z.ZodOptional<z.ZodNumber>;
            default_max_retries: z.ZodOptional<z.ZodNumber>;
            default_retry_backoff_ms: z.ZodOptional<z.ZodNumber>;
            default_retry_max_backoff_ms: z.ZodOptional<z.ZodNumber>;
            default_retry_multiplier: z.ZodOptional<z.ZodNumber>;
            max_in_flight_per_client: z.ZodOptional<z.ZodNumber>;
            next_offset: z.ZodOptional<z.ZodNumber>;
            message_count: z.ZodOptional<z.ZodNumber>;
            dlq_count: z.ZodOptional<z.ZodNumber>;
            created_at: z.ZodString;
            updated_at: z.ZodString;
            paused_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
            retention_seconds: z.ZodOptional<z.ZodNumber>;
        }, z.core.$strip>;
    }, z.core.$strip>;
}, z.core.$strip>], "success">;
export declare const QueuesListResponseSchema: z.ZodDiscriminatedUnion<[z.ZodObject<{
    success: z.ZodLiteral<false>;
    message: z.ZodString;
    code: z.ZodOptional<z.ZodString>;
}, z.core.$strip>, z.ZodObject<{
    success: z.ZodLiteral<true>;
    data: z.ZodObject<{
        queues: z.ZodArray<z.ZodObject<{
            id: z.ZodString;
            name: z.ZodString;
            description: z.ZodOptional<z.ZodNullable<z.ZodString>>;
            queue_type: z.ZodEnum<{
                worker: "worker";
                pubsub: "pubsub";
            }>;
            default_ttl_seconds: z.ZodOptional<z.ZodNullable<z.ZodNumber>>;
            default_visibility_timeout_seconds: z.ZodOptional<z.ZodNumber>;
            default_max_retries: z.ZodOptional<z.ZodNumber>;
            default_retry_backoff_ms: z.ZodOptional<z.ZodNumber>;
            default_retry_max_backoff_ms: z.ZodOptional<z.ZodNumber>;
            default_retry_multiplier: z.ZodOptional<z.ZodNumber>;
            max_in_flight_per_client: z.ZodOptional<z.ZodNumber>;
            next_offset: z.ZodOptional<z.ZodNumber>;
            message_count: z.ZodOptional<z.ZodNumber>;
            dlq_count: z.ZodOptional<z.ZodNumber>;
            created_at: z.ZodString;
            updated_at: z.ZodString;
            paused_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
            retention_seconds: z.ZodOptional<z.ZodNumber>;
        }, z.core.$strip>>;
        total: z.ZodOptional<z.ZodNumber>;
    }, z.core.$strip>;
}, z.core.$strip>], "success">;
export declare const DeleteQueueResponseSchema: z.ZodDiscriminatedUnion<[z.ZodObject<{
    success: z.ZodLiteral<false>;
    message: z.ZodString;
    code: z.ZodOptional<z.ZodString>;
}, z.core.$strip>, z.ZodObject<{
    success: z.ZodLiteral<true>;
}, z.core.$strip>], "success">;
/**
 * Create a new message queue.
 *
 * Creates a queue with the specified type and settings. The queue name is optional
 * and will be auto-generated if not provided.
 *
 * @param client - The API client instance
 * @param params - Queue creation parameters
 * @returns The created queue
 * @throws {QueueValidationError} If validation fails
 * @throws {QueueError} If the API request fails
 *
 * @example
 * ```typescript
 * // Create a worker queue with custom settings
 * const queue = await createQueue(client, {
 *   name: 'order-processing',
 *   queue_type: 'worker',
 *   description: 'Processes customer orders',
 *   settings: {
 *     default_max_retries: 3,
 *     default_visibility_timeout_seconds: 60,
 *   },
 * });
 * console.log(`Created queue: ${queue.id}`);
 * ```
 */
export declare function createQueue(client: APIClient, params: CreateQueueRequest, options?: QueueApiOptions): Promise<Queue>;
/**
 * Get a queue by name.
 *
 * Retrieves the queue details including settings and statistics.
 *
 * @param client - The API client instance
 * @param name - The queue name
 * @returns The queue details
 * @throws {QueueValidationError} If the queue name is invalid
 * @throws {QueueNotFoundError} If the queue does not exist
 * @throws {QueueError} If the API request fails
 *
 * @example
 * ```typescript
 * const queue = await getQueue(client, 'order-processing');
 * console.log(`Queue has ${queue.stats?.message_count} messages`);
 * ```
 */
export declare function getQueue(client: APIClient, name: string, options?: QueueApiOptions): Promise<Queue>;
/**
 * List all queues with optional pagination.
 *
 * @param client - The API client instance
 * @param params - Optional pagination parameters
 * @returns Object containing the list of queues and optional total count
 * @throws {QueueValidationError} If pagination parameters are invalid
 * @throws {QueueError} If the API request fails
 *
 * @example
 * ```typescript
 * // List first 10 queues
 * const { queues, total } = await listQueues(client, { limit: 10 });
 * console.log(`Found ${total} queues`);
 *
 * // Paginate through all queues
 * const { queues: page2 } = await listQueues(client, { limit: 10, offset: 10 });
 * ```
 */
export declare function listQueues(client: APIClient, params?: ListQueuesRequest, options?: QueueApiOptions): Promise<{
    queues: Queue[];
    total?: number;
}>;
/**
 * Update an existing queue.
 *
 * Updates the queue description and/or settings. Only provided fields are updated.
 *
 * @param client - The API client instance
 * @param name - The queue name
 * @param params - Update parameters
 * @returns The updated queue
 * @throws {QueueValidationError} If validation fails
 * @throws {QueueNotFoundError} If the queue does not exist
 * @throws {QueueError} If the API request fails
 *
 * @example
 * ```typescript
 * const queue = await updateQueue(client, 'order-processing', {
 *   description: 'Updated description',
 *   settings: { default_max_retries: 5 },
 * });
 * ```
 */
export declare function updateQueue(client: APIClient, name: string, params: UpdateQueueRequest, options?: QueueApiOptions): Promise<Queue>;
/**
 * Delete a queue.
 *
 * Permanently deletes a queue and all its messages. This action cannot be undone.
 *
 * @param client - The API client instance
 * @param name - The queue name
 * @throws {QueueValidationError} If the queue name is invalid
 * @throws {QueueNotFoundError} If the queue does not exist
 * @throws {QueueError} If the API request fails
 *
 * @example
 * ```typescript
 * await deleteQueue(client, 'order-processing');
 * console.log('Queue deleted');
 * ```
 */
export declare function deleteQueue(client: APIClient, name: string, options?: QueueApiOptions): Promise<void>;
/**
 * Pause a queue.
 *
 * Pauses message processing for the queue. Messages can still be published
 * but will not be delivered to consumers until the queue is resumed.
 *
 * @param client - The API client instance
 * @param name - The queue name
 * @returns The updated queue with paused_at timestamp
 * @throws {QueueValidationError} If the queue name is invalid
 * @throws {QueueNotFoundError} If the queue does not exist
 * @throws {QueueError} If the API request fails
 *
 * @example
 * ```typescript
 * const queue = await pauseQueue(client, 'order-processing');
 * console.log(`Queue paused at: ${queue.paused_at}`);
 * ```
 */
export declare function pauseQueue(client: APIClient, name: string, options?: QueueApiOptions): Promise<Queue>;
/**
 * Resume a paused queue.
 *
 * Resumes message processing for a paused queue. Consumers will start
 * receiving messages again.
 *
 * @param client - The API client instance
 * @param name - The queue name
 * @returns The updated queue with paused_at cleared
 * @throws {QueueValidationError} If the queue name is invalid
 * @throws {QueueNotFoundError} If the queue does not exist
 * @throws {QueueError} If the API request fails
 *
 * @example
 * ```typescript
 * const queue = await resumeQueue(client, 'order-processing');
 * console.log(`Queue resumed, paused_at: ${queue.paused_at}`); // null
 * ```
 */
export declare function resumeQueue(client: APIClient, name: string, options?: QueueApiOptions): Promise<Queue>;
//# sourceMappingURL=queues.d.ts.map