/**
 * Middleware factories for Vite-native architecture
 * Extracted from _server.ts to be used by generated entry files
 */
import { cors } from 'hono/cors';
import type { Env, CompressionConfig, CorsConfig } from './app';
import type { Logger } from './logger';
import type { Meter, Tracer } from '@opentelemetry/api';
export declare const AGENT_CONTEXT_PROPERTIES: readonly ["logger", "tracer", "sessionId", "kv", "stream", "vector", "sandbox", "queue", "email", "schedule", "task", "state", "thread", "session", "config", "app"];
export interface MiddlewareConfig {
    logger: Logger;
    tracer: Tracer;
    meter: Meter;
    corsOptions?: Parameters<typeof cors>[0];
}
/**
 * Create base middleware that sets up context variables
 */
export declare function createBaseMiddleware(config: MiddlewareConfig): import("hono").MiddlewareHandler<Env<any>, string, {}, Response>;
/**
 * Create CORS middleware with lazy config resolution.
 *
 * Handles Cross-Origin Resource Sharing (CORS) headers for API routes.
 * Config is resolved at request time, allowing it to be set via createApp().
 * Static options passed here take precedence over app config.
 *
 * Default behavior:
 * - Reflects the request origin (allows any origin)
 * - Allows common headers: Content-Type, Authorization, Accept, Origin, X-Requested-With
 * - Allows all standard HTTP methods
 * - Enables credentials
 * - Sets max-age to 600 seconds (10 minutes)
 *
 * @param staticOptions - Optional static CORS options that override app config
 *
 * @example
 * ```typescript
 * // Use with default settings
 * app.use('/api/*', createCorsMiddleware());
 *
 * // Or configure via createApp
 * const app = await createApp({
 *   cors: {
 *     origin: 'https://example.com',
 *     allowHeaders: ['Content-Type', 'Authorization', 'X-Custom-Header'],
 *     maxAge: 3600,
 *   }
 * });
 *
 * // Or pass static options directly (overrides app config)
 * app.use('/api/*', createCorsMiddleware({
 *   origin: ['https://app.example.com', 'https://admin.example.com'],
 *   credentials: true,
 * }));
 * ```
 */
export declare function createCorsMiddleware(staticOptions?: CorsConfig): import("hono").MiddlewareHandler<Env<any>, string, {}, Response>;
/**
 * Create OpenTelemetry middleware for session/thread tracking
 * This is the critical middleware that creates AgentContext
 */
export declare function createOtelMiddleware(): import("hono").MiddlewareHandler<Env<any>, string, {}, Response>;
/**
 * Create compression middleware with lazy config resolution.
 *
 * Compresses response bodies using gzip or deflate based on the Accept-Encoding header.
 * Config is resolved at request time, allowing it to be set via createApp().
 *
 * @param staticConfig - Optional static config that overrides app config
 *
 * @example
 * ```typescript
 * // Use with default settings
 * app.use('*', createCompressionMiddleware());
 *
 * // Or configure via createApp
 * const app = await createApp({
 *   compression: {
 *     threshold: 2048,
 *   }
 * });
 * ```
 */
export declare function createCompressionMiddleware(staticConfig?: CompressionConfig, 
/**
 * Optional config resolver for testing. When provided, this is used instead of getAppConfig().
 * @internal
 */
configResolver?: () => {
    compression?: CompressionConfig | false;
} | undefined): import("hono").MiddlewareHandler<Env<any>, string, {}, Response>;
/**
 * Create lightweight thread middleware for web routes (analytics).
 *
 * Sets thread cookie that persists across page views for client-side analytics.
 * This middleware does NOT:
 * - Create or track sessions (no session ID)
 * - Set session/thread response headers
 * - Send events to Catalyst sessions table
 *
 * This is intentionally separate from createOtelMiddleware to avoid
 * polluting the sessions table with web browsing activity.
 *
 * - Thread cookie (atid_a): Analytics-readable copy, 1-week expiry
 */
export declare function createWebSessionMiddleware(): import("hono").MiddlewareHandler<Env<any>, string, {}, Response>;
/**
 * Get the secret used for signing session/thread cookies.
 * Uses AGENTUITY_SDK_KEY if available, falls back to 'agentuity'.
 */
export declare function getSessionSecret(): string;
//# sourceMappingURL=middleware.d.ts.map