import type { Context, MiddlewareHandler } from 'hono';
import type { Env } from '../app';
/**
 * Context key for WebSocket close promise.
 * Used by middleware to defer session finalization until WebSocket closes.
 */
export declare const WS_DONE_PROMISE_KEY = "_wsDonePromise";
/**
 * WebSocket connection interface for handling WebSocket events.
 */
export interface WebSocketConnection {
    onOpen: (handler: (event: Event) => void | Promise<void>) => void;
    onMessage: (handler: (event: MessageEvent) => void | Promise<void>) => void;
    onClose: (handler: (event: CloseEvent) => void | Promise<void>) => void;
    send: (data: string | ArrayBuffer | Uint8Array) => void;
}
/**
 * Handler function for WebSocket connections.
 * Receives the Hono context and WebSocket connection with a flattened signature.
 *
 * **This handler must be synchronous** (returns `void`, not `Promise<void>`).
 * The handler is called inside Hono's `upgradeWebSocket` factory, which must
 * return event handlers synchronously for the HTTP upgrade to complete. If the
 * handler were async, any `ws.onOpen`/`ws.onMessage`/`ws.onClose` registrations
 * after an `await` would be silently lost because the factory returns before
 * they are registered.
 *
 * To perform async work, place it inside the `onOpen`, `onMessage`, or `onClose`
 * callbacks, which are properly awaited by the runtime.
 */
export type WebSocketHandler<E extends Env = Env> = (c: Context<E>, ws: WebSocketConnection) => void;
/**
 * Creates a WebSocket middleware for handling WebSocket connections.
 *
 * The handler must be **synchronous** — it runs inside Hono's `upgradeWebSocket`
 * factory which must return event handlers synchronously for the HTTP upgrade to
 * complete. Async work should go inside `onOpen`, `onMessage`, or `onClose`
 * callbacks, which are properly awaited by the runtime.
 *
 * Use with router.get() to create a WebSocket endpoint:
 *
 * @example
 * ```typescript
 * // Basic synchronous usage
 * import { createRouter, websocket } from '@agentuity/runtime';
 *
 * const router = createRouter();
 *
 * router.get('/ws', websocket((c, ws) => {
 *   ws.onOpen(() => {
 *     c.var.logger.info('WebSocket opened');
 *     ws.send('Welcome!');
 *   });
 *
 *   ws.onMessage((event) => {
 *     c.var.logger.info('Received:', event.data);
 *     ws.send('Echo: ' + event.data);
 *   });
 *
 *   ws.onClose(() => {
 *     c.var.logger.info('WebSocket closed');
 *   });
 * }));
 * ```
 *
 * @example
 * ```typescript
 * // Async work inside callbacks (correct pattern)
 * router.get('/ws', websocket((c, ws) => {
 *   ws.onOpen(async () => {
 *     const user = await fetchUser(c.var.auth);
 *     ws.send(JSON.stringify({ welcome: user.name }));
 *   });
 *
 *   ws.onMessage(async (event) => {
 *     const result = await processMessage(event.data);
 *     ws.send(JSON.stringify(result));
 *   });
 * }));
 * ```
 *
 * @param handler - Synchronous handler function receiving context and WebSocket connection
 * @returns Hono middleware handler for WebSocket upgrade
 */
export declare function websocket<E extends Env = Env>(handler: WebSocketHandler<E>): MiddlewareHandler<E>;
//# sourceMappingURL=websocket.d.ts.map