import type { PluginInput } from '@opencode-ai/plugin';
import type { AgentConfig, CoderConfig } from '../../types';
export interface ParamsHooks {
    onParams: (input: unknown, output: unknown) => Promise<void>;
}
export declare function createParamsHooks(ctx: PluginInput, _config: CoderConfig, lastUserMessages?: Map<string, string>): ParamsHooks;
/**
 * Advertised magic words for users:
 *
 * - "brainstorm" - Activates creative mode (temperature → 0.8)
 * - "think hard" - Activates deep thinking mode (max reasoning budget)
 * - "dig deep" / "go deep" - Activates thorough mode (maxSteps → 50)
 *
 * These can also be triggered by specific phrases like:
 * - "be creative", "give me ideas", "explore options", "explore alternatives"
 * - "think deeply", "analyze carefully", "reason through this"
 * - "deep dive", "take your time", "be thorough", "be meticulous"
 *
 * Note: Triggers use multi-word phrases to avoid false positives from common words.
 */
/** Retryable HTTP status codes that should trigger model fallback */
export declare const RETRYABLE_STATUS_CODES: readonly [429, 500, 502, 503];
/**
 * Tracks API errors per agent to enable model fallback on subsequent calls.
 *
 * When an agent's primary model fails with a retryable error (429, 500, 502, 503),
 * the next `chat.params` call can select a fallback model from the agent's
 * `fallbackModels` list.
 *
 * Current limitation: The `chat.params` hook can modify temperature/topP/topK/options
 * but CANNOT change the model itself (model is in the input, not output). Full model
 * fallback requires one of:
 *   1. A `chat.error` hook that allows retrying with a different model
 *   2. A `chat.model` hook that allows overriding the model selection
 *   3. Adding `model` to the `chat.params` output type
 *
 * TODO: When OpenCode adds a suitable hook, implement the retry logic here:
 *   - On API error (429/5xx), record the failure in `agentErrorState`
 *   - On next `chat.params` call for the same agent, select next fallback model
 *   - Log: `[ModelFallback] Switching from ${currentModel} to ${fallbackModel} due to ${error}`
 *   - Reset fallback state after successful completion or after TTL expires
 */
export declare class ModelFallbackTracker {
    /**
     * Map of agent name → { failedModel, failedAt, errorCode, fallbackIndex }
     * Used to track which agents have experienced API errors.
     */
    private agentErrorState;
    /** TTL for error state — reset after 5 minutes */
    private readonly ERROR_STATE_TTL_MS;
    /**
     * Record an API error for an agent. Call this from an event handler
     * when a retryable API error is detected.
     */
    recordError(agentName: string, model: string, errorCode: number): void;
    /**
     * Get the next fallback model for an agent, if one is available.
     * Returns undefined if no fallback is needed or available.
     */
    getNextFallback(agentName: string, agentConfig: AgentConfig): string | undefined;
    /**
     * Clear error state for an agent (e.g., after successful completion).
     */
    clearError(agentName: string): void;
}
//# sourceMappingURL=params.d.ts.map