import { buildUrl, toServiceException } from "./_util.js";
import { StructuredError } from "../error.js";
import { safeStringify } from "../json.js";
const TASK_API_VERSION = '2026-02-24';
const TASK_ACTIVITY_API_VERSION = '2026-02-28';
const TaskIdRequiredError = StructuredError('TaskIdRequiredError', 'Task ID is required and must be a non-empty string');
const TaskTitleRequiredError = StructuredError('TaskTitleRequiredError', 'Task title is required and must be a non-empty string');
const CommentIdRequiredError = StructuredError('CommentIdRequiredError', 'Comment ID is required and must be a non-empty string');
const CommentBodyRequiredError = StructuredError('CommentBodyRequiredError', 'Comment body is required and must be a non-empty string');
const TagIdRequiredError = StructuredError('TagIdRequiredError', 'Tag ID is required and must be a non-empty string');
const TagNameRequiredError = StructuredError('TagNameRequiredError', 'Tag name is required and must be a non-empty string');
const AttachmentIdRequiredError = StructuredError('AttachmentIdRequiredError', 'Attachment ID is required and must be a non-empty string');
const UserIdRequiredError = StructuredError('UserIdRequiredError', 'User ID is required and must be a non-empty string');
const TaskStorageResponseError = StructuredError('TaskStorageResponseError')();
export class TaskStorageService {
    #adapter;
    #baseUrl;
    constructor(baseUrl, adapter) {
        this.#adapter = adapter;
        this.#baseUrl = baseUrl;
    }
    async create(params) {
        if (!params?.title || typeof params.title !== 'string' || params.title.trim().length === 0) {
            throw new TaskTitleRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/${TASK_API_VERSION}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'POST',
            body: safeStringify(params),
            contentType: 'application/json',
            signal,
            telemetry: {
                name: 'agentuity.task.create',
                attributes: {
                    type: params.type,
                    priority: params.priority ?? 'none',
                    status: params.status ?? 'open',
                },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('POST', url, res.response);
    }
    async get(id) {
        if (!id || typeof id !== 'string' || id.trim().length === 0) {
            throw new TaskIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/${TASK_API_VERSION}/${encodeURIComponent(id)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.task.get',
                attributes: { id },
            },
        });
        if (res.response.status === 404) {
            return null;
        }
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('GET', url, res.response);
    }
    async list(params) {
        const queryParams = new URLSearchParams();
        if (params?.status)
            queryParams.set('status', params.status);
        if (params?.type)
            queryParams.set('type', params.type);
        if (params?.priority)
            queryParams.set('priority', params.priority);
        if (params?.assigned_id)
            queryParams.set('assigned_id', params.assigned_id);
        if (params?.parent_id)
            queryParams.set('parent_id', params.parent_id);
        if (params?.project_id)
            queryParams.set('project_id', params.project_id);
        if (params?.tag_id)
            queryParams.set('tag_id', params.tag_id);
        if (params?.deleted !== undefined)
            queryParams.set('deleted', String(params.deleted));
        if (params?.sort)
            queryParams.set('sort', params.sort);
        if (params?.order)
            queryParams.set('order', params.order);
        if (params?.limit !== undefined)
            queryParams.set('limit', String(params.limit));
        if (params?.offset !== undefined)
            queryParams.set('offset', String(params.offset));
        const queryString = queryParams.toString();
        const url = buildUrl(this.#baseUrl, `/task/${TASK_API_VERSION}${queryString ? `?${queryString}` : ''}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.task.list',
                attributes: {
                    ...(params?.status ? { status: params.status } : {}),
                    ...(params?.type ? { type: params.type } : {}),
                    ...(params?.priority ? { priority: params.priority } : {}),
                },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('GET', url, res.response);
    }
    async update(id, params) {
        if (!id || typeof id !== 'string' || id.trim().length === 0) {
            throw new TaskIdRequiredError();
        }
        if (params.title !== undefined &&
            (typeof params.title !== 'string' || params.title.trim().length === 0)) {
            throw new TaskTitleRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/${TASK_API_VERSION}/${encodeURIComponent(id)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'PATCH',
            body: safeStringify(params),
            contentType: 'application/json',
            signal,
            telemetry: {
                name: 'agentuity.task.update',
                attributes: { id },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('PATCH', url, res.response);
    }
    async close(id) {
        if (!id || typeof id !== 'string' || id.trim().length === 0) {
            throw new TaskIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/${TASK_API_VERSION}/${encodeURIComponent(id)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'DELETE',
            signal,
            telemetry: {
                name: 'agentuity.task.close',
                attributes: { id },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('DELETE', url, res.response);
    }
    async changelog(id, params) {
        if (!id || typeof id !== 'string' || id.trim().length === 0) {
            throw new TaskIdRequiredError();
        }
        const queryParams = new URLSearchParams();
        if (params?.limit !== undefined)
            queryParams.set('limit', String(params.limit));
        if (params?.offset !== undefined)
            queryParams.set('offset', String(params.offset));
        const queryString = queryParams.toString();
        const url = buildUrl(this.#baseUrl, `/task/changelog/${TASK_API_VERSION}/${encodeURIComponent(id)}${queryString ? `?${queryString}` : ''}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.task.changelog',
                attributes: { id },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('GET', url, res.response);
    }
    async softDelete(id) {
        if (!id || typeof id !== 'string' || id.trim().length === 0) {
            throw new TaskIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/delete/${TASK_API_VERSION}/${encodeURIComponent(id)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'POST',
            signal,
            telemetry: {
                name: 'agentuity.task.softDelete',
                attributes: { id },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('POST', url, res.response);
    }
    async createComment(taskId, body, userId, author) {
        if (!taskId || typeof taskId !== 'string' || taskId.trim().length === 0) {
            throw new TaskIdRequiredError();
        }
        if (!body || typeof body !== 'string' || body.trim().length === 0) {
            throw new CommentBodyRequiredError();
        }
        if (!userId || typeof userId !== 'string' || userId.trim().length === 0) {
            throw new UserIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/comments/create/${TASK_API_VERSION}/${encodeURIComponent(taskId)}`);
        const signal = AbortSignal.timeout(30_000);
        const commentBody = { body, user_id: userId };
        if (author)
            commentBody.author = author;
        const res = await this.#adapter.invoke(url, {
            method: 'POST',
            body: safeStringify(commentBody),
            contentType: 'application/json',
            signal,
            telemetry: {
                name: 'agentuity.task.createComment',
                attributes: { taskId },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('POST', url, res.response);
    }
    async getComment(commentId) {
        if (!commentId || typeof commentId !== 'string' || commentId.trim().length === 0) {
            throw new CommentIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/comments/get/${TASK_API_VERSION}/${encodeURIComponent(commentId)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.task.getComment',
                attributes: { commentId },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('GET', url, res.response);
    }
    async updateComment(commentId, body) {
        if (!commentId || typeof commentId !== 'string' || commentId.trim().length === 0) {
            throw new CommentIdRequiredError();
        }
        if (!body || typeof body !== 'string' || body.trim().length === 0) {
            throw new CommentBodyRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/comments/update/${TASK_API_VERSION}/${encodeURIComponent(commentId)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'PATCH',
            body: safeStringify({ body }),
            contentType: 'application/json',
            signal,
            telemetry: {
                name: 'agentuity.task.updateComment',
                attributes: { commentId },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('PATCH', url, res.response);
    }
    async deleteComment(commentId) {
        if (!commentId || typeof commentId !== 'string' || commentId.trim().length === 0) {
            throw new CommentIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/comments/delete/${TASK_API_VERSION}/${encodeURIComponent(commentId)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'DELETE',
            signal,
            telemetry: {
                name: 'agentuity.task.deleteComment',
                attributes: { commentId },
            },
        });
        if (res.ok) {
            if (res.data?.success === false) {
                throw new TaskStorageResponseError({
                    status: res.response.status,
                    message: res.data.message ?? 'Operation failed',
                });
            }
            return;
        }
        throw await toServiceException('DELETE', url, res.response);
    }
    async listComments(taskId, params) {
        if (!taskId || typeof taskId !== 'string' || taskId.trim().length === 0) {
            throw new TaskIdRequiredError();
        }
        const queryParams = new URLSearchParams();
        if (params?.limit !== undefined)
            queryParams.set('limit', String(params.limit));
        if (params?.offset !== undefined)
            queryParams.set('offset', String(params.offset));
        const queryString = queryParams.toString();
        const url = buildUrl(this.#baseUrl, `/task/comments/list/${TASK_API_VERSION}/${encodeURIComponent(taskId)}${queryString ? `?${queryString}` : ''}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.task.listComments',
                attributes: { taskId },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('GET', url, res.response);
    }
    async createTag(name, color) {
        if (!name || typeof name !== 'string' || name.trim().length === 0) {
            throw new TagNameRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/tags/create/${TASK_API_VERSION}`);
        const signal = AbortSignal.timeout(30_000);
        const body = { name };
        if (color !== undefined)
            body.color = color;
        const res = await this.#adapter.invoke(url, {
            method: 'POST',
            body: safeStringify(body),
            contentType: 'application/json',
            signal,
            telemetry: {
                name: 'agentuity.task.createTag',
                attributes: { tagName: name },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('POST', url, res.response);
    }
    async getTag(tagId) {
        if (!tagId || typeof tagId !== 'string' || tagId.trim().length === 0) {
            throw new TagIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/tags/get/${TASK_API_VERSION}/${encodeURIComponent(tagId)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.task.getTag',
                attributes: { tagId },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('GET', url, res.response);
    }
    async updateTag(tagId, name, color) {
        if (!tagId || typeof tagId !== 'string' || tagId.trim().length === 0) {
            throw new TagIdRequiredError();
        }
        if (!name || typeof name !== 'string' || name.trim().length === 0) {
            throw new TagNameRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/tags/update/${TASK_API_VERSION}/${encodeURIComponent(tagId)}`);
        const signal = AbortSignal.timeout(30_000);
        const body = { name };
        if (color !== undefined)
            body.color = color;
        const res = await this.#adapter.invoke(url, {
            method: 'PATCH',
            body: safeStringify(body),
            contentType: 'application/json',
            signal,
            telemetry: {
                name: 'agentuity.task.updateTag',
                attributes: { tagId },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('PATCH', url, res.response);
    }
    async deleteTag(tagId) {
        if (!tagId || typeof tagId !== 'string' || tagId.trim().length === 0) {
            throw new TagIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/tags/delete/${TASK_API_VERSION}/${encodeURIComponent(tagId)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'DELETE',
            signal,
            telemetry: {
                name: 'agentuity.task.deleteTag',
                attributes: { tagId },
            },
        });
        if (res.ok) {
            if (res.data?.success === false) {
                throw new TaskStorageResponseError({
                    status: res.response.status,
                    message: res.data.message ?? 'Operation failed',
                });
            }
            return;
        }
        throw await toServiceException('DELETE', url, res.response);
    }
    async listTags() {
        const url = buildUrl(this.#baseUrl, `/task/tags/list/${TASK_API_VERSION}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.task.listTags',
                attributes: {},
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('GET', url, res.response);
    }
    async addTagToTask(taskId, tagId) {
        if (!taskId || typeof taskId !== 'string' || taskId.trim().length === 0) {
            throw new TaskIdRequiredError();
        }
        if (!tagId || typeof tagId !== 'string' || tagId.trim().length === 0) {
            throw new TagIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/tags/add/${TASK_API_VERSION}/${encodeURIComponent(taskId)}/${encodeURIComponent(tagId)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'POST',
            signal,
            telemetry: {
                name: 'agentuity.task.addTagToTask',
                attributes: { taskId, tagId },
            },
        });
        if (res.ok) {
            if (res.data?.success === false) {
                throw new TaskStorageResponseError({
                    status: res.response.status,
                    message: res.data.message ?? 'Operation failed',
                });
            }
            return;
        }
        throw await toServiceException('POST', url, res.response);
    }
    async removeTagFromTask(taskId, tagId) {
        if (!taskId || typeof taskId !== 'string' || taskId.trim().length === 0) {
            throw new TaskIdRequiredError();
        }
        if (!tagId || typeof tagId !== 'string' || tagId.trim().length === 0) {
            throw new TagIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/tags/remove/${TASK_API_VERSION}/${encodeURIComponent(taskId)}/${encodeURIComponent(tagId)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'DELETE',
            signal,
            telemetry: {
                name: 'agentuity.task.removeTagFromTask',
                attributes: { taskId, tagId },
            },
        });
        if (res.ok) {
            if (res.data?.success === false) {
                throw new TaskStorageResponseError({
                    status: res.response.status,
                    message: res.data.message ?? 'Operation failed',
                });
            }
            return;
        }
        throw await toServiceException('DELETE', url, res.response);
    }
    async listTagsForTask(taskId) {
        if (!taskId || typeof taskId !== 'string' || taskId.trim().length === 0) {
            throw new TaskIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/tags/task/${TASK_API_VERSION}/${encodeURIComponent(taskId)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.task.listTagsForTask',
                attributes: { taskId },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('GET', url, res.response);
    }
    // Attachment methods
    async uploadAttachment(taskId, params) {
        if (!taskId || typeof taskId !== 'string' || taskId.trim().length === 0) {
            throw new TaskIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/attachments/presign-upload/${TASK_API_VERSION}/${encodeURIComponent(taskId)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'POST',
            body: safeStringify(params),
            contentType: 'application/json',
            signal,
            telemetry: {
                name: 'agentuity.task.uploadAttachment',
                attributes: { taskId },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('POST', url, res.response);
    }
    async confirmAttachment(attachmentId) {
        if (!attachmentId || typeof attachmentId !== 'string' || attachmentId.trim().length === 0) {
            throw new AttachmentIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/attachments/confirm/${TASK_API_VERSION}/${encodeURIComponent(attachmentId)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'POST',
            signal,
            telemetry: {
                name: 'agentuity.task.confirmAttachment',
                attributes: { attachmentId },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('POST', url, res.response);
    }
    async downloadAttachment(attachmentId) {
        if (!attachmentId || typeof attachmentId !== 'string' || attachmentId.trim().length === 0) {
            throw new AttachmentIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/attachments/presign-download/${TASK_API_VERSION}/${encodeURIComponent(attachmentId)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'POST',
            signal,
            telemetry: {
                name: 'agentuity.task.downloadAttachment',
                attributes: { attachmentId },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('POST', url, res.response);
    }
    async listAttachments(taskId) {
        if (!taskId || typeof taskId !== 'string' || taskId.trim().length === 0) {
            throw new TaskIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/attachments/list/${TASK_API_VERSION}/${encodeURIComponent(taskId)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.task.listAttachments',
                attributes: { taskId },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('GET', url, res.response);
    }
    async deleteAttachment(attachmentId) {
        if (!attachmentId || typeof attachmentId !== 'string' || attachmentId.trim().length === 0) {
            throw new AttachmentIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/attachments/delete/${TASK_API_VERSION}/${encodeURIComponent(attachmentId)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'DELETE',
            signal,
            telemetry: {
                name: 'agentuity.task.deleteAttachment',
                attributes: { attachmentId },
            },
        });
        if (res.ok) {
            if (res.data?.success === false) {
                throw new TaskStorageResponseError({
                    status: res.response.status,
                    message: res.data.message ?? 'Operation failed',
                });
            }
            return;
        }
        throw await toServiceException('DELETE', url, res.response);
    }
    async listUsers() {
        const url = buildUrl(this.#baseUrl, `/task/users/${TASK_API_VERSION}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.task.listUsers',
                attributes: {},
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('GET', url, res.response);
    }
    async listProjects() {
        const url = buildUrl(this.#baseUrl, `/task/projects/${TASK_API_VERSION}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.task.listProjects',
                attributes: {},
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('GET', url, res.response);
    }
    async getActivity(params) {
        const queryParams = new URLSearchParams();
        if (params?.days !== undefined)
            queryParams.set('days', String(params.days));
        const queryString = queryParams.toString();
        const url = buildUrl(this.#baseUrl, `/task/activity/${TASK_ACTIVITY_API_VERSION}${queryString ? `?${queryString}` : ''}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.task.activity',
                attributes: {
                    ...(params?.days !== undefined ? { days: String(params.days) } : {}),
                },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('GET', url, res.response);
    }
}
//# sourceMappingURL=task.js.map