import { FetchAdapter } from './adapter.ts';
export type TaskPriority = 'high' | 'medium' | 'low' | 'none';
export type TaskType = 'epic' | 'feature' | 'enhancement' | 'bug' | 'task';
export type TaskStatus = 'open' | 'in_progress' | 'closed' | 'done' | 'cancelled';
export interface EntityRef {
    id: string;
    name: string;
}
export interface Task {
    id: string;
    created_at: string;
    updated_at: string;
    title: string;
    description?: string;
    metadata?: Record<string, unknown>;
    priority: TaskPriority;
    parent_id?: string;
    type: TaskType;
    status: TaskStatus;
    open_date?: string;
    in_progress_date?: string;
    closed_date?: string;
    created_id: string;
    assigned_id?: string;
    closed_id?: string;
    creator?: EntityRef;
    assignee?: EntityRef;
    closer?: EntityRef;
    project?: EntityRef;
    cancelled_date?: string;
    deleted: boolean;
    tags?: Tag[];
    comments?: Comment[];
}
export interface Comment {
    id: string;
    created_at: string;
    updated_at: string;
    task_id: string;
    user_id: string;
    author?: EntityRef;
    body: string;
}
export interface Tag {
    id: string;
    created_at: string;
    name: string;
    color?: string;
}
export interface TaskChangelogEntry {
    id: string;
    created_at: string;
    task_id: string;
    field: string;
    old_value?: string;
    new_value?: string;
}
export interface CreateTaskParams {
    title: string;
    description?: string;
    metadata?: Record<string, unknown>;
    priority?: TaskPriority;
    parent_id?: string;
    type: TaskType;
    status?: TaskStatus;
    created_id: string;
    assigned_id?: string;
    creator?: EntityRef;
    assignee?: EntityRef;
    project?: EntityRef;
    tag_ids?: string[];
}
export interface UpdateTaskParams {
    title?: string;
    description?: string;
    metadata?: Record<string, unknown>;
    priority?: TaskPriority;
    parent_id?: string;
    type?: TaskType;
    status?: TaskStatus;
    assigned_id?: string;
    closed_id?: string;
    assignee?: EntityRef;
    closer?: EntityRef;
    project?: EntityRef;
}
export interface ListTasksParams {
    status?: TaskStatus;
    type?: TaskType;
    priority?: TaskPriority;
    assigned_id?: string;
    parent_id?: string;
    project_id?: string;
    tag_id?: string;
    deleted?: boolean;
    sort?: string;
    order?: 'asc' | 'desc';
    limit?: number;
    offset?: number;
}
export interface ListTasksResult {
    tasks: Task[];
    total: number;
    limit: number;
    offset: number;
}
export interface TaskChangelogResult {
    changelog: TaskChangelogEntry[];
    total: number;
    limit: number;
    offset: number;
}
export interface ListCommentsResult {
    comments: Comment[];
    total: number;
    limit: number;
    offset: number;
}
export interface ListTagsResult {
    tags: Tag[];
}
export interface Attachment {
    id: string;
    created_at: string;
    task_id: string;
    user_id: string;
    author?: EntityRef;
    filename: string;
    content_type?: string;
    size?: number;
}
export interface CreateAttachmentParams {
    filename: string;
    content_type?: string;
    size?: number;
}
export interface PresignUploadResponse {
    attachment: Attachment;
    presigned_url: string;
    expiry_seconds: number;
}
export interface PresignDownloadResponse {
    presigned_url: string;
    expiry_seconds: number;
}
export interface ListAttachmentsResult {
    attachments: Attachment[];
    total: number;
}
export interface ListUsersResult {
    users: EntityRef[];
}
export interface ListProjectsResult {
    projects: EntityRef[];
}
export interface TaskActivityParams {
    days?: number;
}
export interface TaskActivityDataPoint {
    date: string;
    open: number;
    inProgress: number;
    done: number;
    closed: number;
    cancelled: number;
}
export interface TaskActivityResult {
    activity: TaskActivityDataPoint[];
    days: number;
}
export interface TaskStorage {
    create(params: CreateTaskParams): Promise<Task>;
    get(id: string): Promise<Task | null>;
    list(params?: ListTasksParams): Promise<ListTasksResult>;
    update(id: string, params: UpdateTaskParams): Promise<Task>;
    close(id: string): Promise<Task>;
    softDelete(id: string): Promise<Task>;
    changelog(id: string, params?: {
        limit?: number;
        offset?: number;
    }): Promise<TaskChangelogResult>;
    createComment(taskId: string, body: string, userId: string, author?: EntityRef): Promise<Comment>;
    getComment(commentId: string): Promise<Comment>;
    updateComment(commentId: string, body: string): Promise<Comment>;
    deleteComment(commentId: string): Promise<void>;
    listComments(taskId: string, params?: {
        limit?: number;
        offset?: number;
    }): Promise<ListCommentsResult>;
    createTag(name: string, color?: string): Promise<Tag>;
    getTag(tagId: string): Promise<Tag>;
    updateTag(tagId: string, name: string, color?: string): Promise<Tag>;
    deleteTag(tagId: string): Promise<void>;
    listTags(): Promise<ListTagsResult>;
    addTagToTask(taskId: string, tagId: string): Promise<void>;
    removeTagFromTask(taskId: string, tagId: string): Promise<void>;
    listTagsForTask(taskId: string): Promise<Tag[]>;
    uploadAttachment(taskId: string, params: CreateAttachmentParams): Promise<PresignUploadResponse>;
    confirmAttachment(attachmentId: string): Promise<Attachment>;
    downloadAttachment(attachmentId: string): Promise<PresignDownloadResponse>;
    listAttachments(taskId: string): Promise<ListAttachmentsResult>;
    deleteAttachment(attachmentId: string): Promise<void>;
    listUsers(): Promise<ListUsersResult>;
    listProjects(): Promise<ListProjectsResult>;
    getActivity(params?: TaskActivityParams): Promise<TaskActivityResult>;
}
export declare class TaskStorageService implements TaskStorage {
    #private;
    constructor(baseUrl: string, adapter: FetchAdapter);
    create(params: CreateTaskParams): Promise<Task>;
    get(id: string): Promise<Task | null>;
    list(params?: ListTasksParams): Promise<ListTasksResult>;
    update(id: string, params: UpdateTaskParams): Promise<Task>;
    close(id: string): Promise<Task>;
    changelog(id: string, params?: {
        limit?: number;
        offset?: number;
    }): Promise<TaskChangelogResult>;
    softDelete(id: string): Promise<Task>;
    createComment(taskId: string, body: string, userId: string, author?: EntityRef): Promise<Comment>;
    getComment(commentId: string): Promise<Comment>;
    updateComment(commentId: string, body: string): Promise<Comment>;
    deleteComment(commentId: string): Promise<void>;
    listComments(taskId: string, params?: {
        limit?: number;
        offset?: number;
    }): Promise<ListCommentsResult>;
    createTag(name: string, color?: string): Promise<Tag>;
    getTag(tagId: string): Promise<Tag>;
    updateTag(tagId: string, name: string, color?: string): Promise<Tag>;
    deleteTag(tagId: string): Promise<void>;
    listTags(): Promise<ListTagsResult>;
    addTagToTask(taskId: string, tagId: string): Promise<void>;
    removeTagFromTask(taskId: string, tagId: string): Promise<void>;
    listTagsForTask(taskId: string): Promise<Tag[]>;
    uploadAttachment(taskId: string, params: CreateAttachmentParams): Promise<PresignUploadResponse>;
    confirmAttachment(attachmentId: string): Promise<Attachment>;
    downloadAttachment(attachmentId: string): Promise<PresignDownloadResponse>;
    listAttachments(taskId: string): Promise<ListAttachmentsResult>;
    deleteAttachment(attachmentId: string): Promise<void>;
    listUsers(): Promise<ListUsersResult>;
    listProjects(): Promise<ListProjectsResult>;
    getActivity(params?: TaskActivityParams): Promise<TaskActivityResult>;
}
//# sourceMappingURL=task.d.ts.map