import { FetchAdapter } from './adapter.ts';
/**
 * An email address registered with the Agentuity email service
 */
export interface EmailAddress {
    id: string;
    email: string;
    config?: Record<string, unknown>;
    created_by?: string;
    created_at: string;
    updated_at?: string;
    inbound_count?: number;
    outbound_count?: number;
    last_activity?: string;
}
/**
 * A destination configuration for an email address
 */
export interface EmailDestination {
    id: string;
    type: string;
    config?: Record<string, unknown>;
    created_at: string;
    updated_at?: string;
}
export interface EmailProtocolConfig {
    host: string;
    port: number;
    tls: string;
    username: string;
    password: string;
}
export interface EmailConnectionConfig {
    email: string;
    imap: EmailProtocolConfig;
    pop3: EmailProtocolConfig;
}
/**
 * An inbound email message
 */
export interface EmailInbound {
    id: string;
    from: string;
    to: string;
    subject?: string;
    text?: string;
    html?: string;
    received_at?: string;
    headers?: Record<string, unknown>;
    attachments?: EmailStoredAttachment[];
}
/**
 * An outbound email message
 */
export interface EmailOutbound {
    id: string;
    from: string;
    to: string;
    subject?: string;
    text?: string;
    html?: string;
    status?: string;
    error?: string;
    created_at?: string;
    headers?: Record<string, unknown>;
    attachments?: EmailStoredAttachment[];
}
/**
 * An email attachment
 */
export interface EmailAttachment {
    /**
     * The filename for the attachment
     */
    filename: string;
    /**
     * The base64-encoded content of the attachment
     */
    content: string;
    /**
     * The MIME content type of the attachment
     */
    contentType?: string;
}
/**
 * A stored email attachment with S3 location metadata.
 * Returned by inbound/outbound email queries — different from EmailAttachment used for sending.
 */
export interface EmailStoredAttachment {
    /** The original filename */
    filename: string;
    /** The MIME content type */
    content_type?: string;
    /** File size in bytes */
    size: number;
    /** The S3 bucket name where the attachment is stored */
    bucket: string;
    /** The S3 object key */
    key: string;
    /** Optional pre-signed download URL */
    url?: string;
}
/**
 * Parameters for sending an email
 */
export interface EmailSendParams {
    /**
     * The sender email address (must be owned by the organization)
     */
    from: string;
    /**
     * The recipient email addresses
     */
    to: string[];
    /**
     * The email subject
     */
    subject: string;
    /**
     * Plain text email body
     */
    text?: string;
    /**
     * HTML email body
     */
    html?: string;
    /**
     * File attachments
     */
    attachments?: EmailAttachment[];
    /**
     * Custom email headers (e.g., In-Reply-To, References for threading)
     */
    headers?: Record<string, string>;
}
/**
 * Parameters for email activity time-series
 */
export interface EmailActivityParams {
    days?: number;
}
/**
 * A single data point in the email activity time-series
 */
export interface EmailActivityDataPoint {
    date: string;
    inbound: number;
    outbound: number;
}
/**
 * Result of email activity query
 */
export interface EmailActivityResult {
    activity: EmailActivityDataPoint[];
    days: number;
}
/**
 * Email service for managing email addresses, destinations, and sending/receiving emails
 */
export interface EmailService {
    /**
     * Create a new email address
     *
     * @param localPart - the local part of the email address (before the @)
     * @returns the created email address
     *
     * @example
     * ```typescript
     * const address = await email.createAddress('support');
     * console.log('Created:', address.email);
     * ```
     */
    createAddress(localPart: string): Promise<EmailAddress>;
    /**
     * List all email addresses
     *
     * @returns array of email addresses
     *
     * @example
     * ```typescript
     * const addresses = await email.listAddresses();
     * for (const addr of addresses) {
     *   console.log(addr.email);
     * }
     * ```
     */
    listAddresses(): Promise<EmailAddress[]>;
    /**
     * Get an email address by ID
     *
     * @param id - the email address ID
     * @returns the email address or null if not found
     *
     * @example
     * ```typescript
     * const address = await email.getAddress('addr_123');
     * if (address) {
     *   console.log('Found:', address.email);
     * }
     * ```
     */
    getAddress(id: string): Promise<EmailAddress | null>;
    /**
     * Get email connection settings (IMAP/POP3) for an address
     *
     * @param id - the email address ID
     * @returns the connection configuration or null if not found
     */
    getConnectionConfig(id: string): Promise<EmailConnectionConfig | null>;
    /**
     * Delete an email address
     *
     * @param id - the email address ID
     *
     * @example
     * ```typescript
     * await email.deleteAddress('addr_123');
     * ```
     */
    deleteAddress(id: string): Promise<void>;
    /**
     * Create a destination for an email address
     *
     * @param addressId - the email address ID
     * @param type - the destination type (e.g., 'url', 'agent')
     * @param config - the destination configuration
     * @returns the created destination
     *
     * @example
     * ```typescript
     * const dest = await email.createDestination('addr_123', 'url', {
     *   url: 'https://example.com/webhook',
     * });
     * console.log('Created destination:', dest.id);
     * ```
     */
    createDestination(addressId: string, type: string, config: Record<string, unknown>): Promise<EmailDestination>;
    /**
     * List destinations for an email address
     *
     * @param addressId - the email address ID
     * @returns array of destinations
     *
     * @example
     * ```typescript
     * const destinations = await email.listDestinations('addr_123');
     * for (const dest of destinations) {
     *   console.log(`${dest.type}: ${dest.id}`);
     * }
     * ```
     */
    listDestinations(addressId: string): Promise<EmailDestination[]>;
    /**
     * Delete a destination from an email address
     *
     * @param addressId - the email address ID
     * @param destinationId - the destination ID
     *
     * @example
     * ```typescript
     * await email.deleteDestination('addr_123', 'dest_456');
     * ```
     */
    deleteDestination(addressId: string, destinationId: string): Promise<void>;
    /**
     * Send an email
     *
     * @param params - the send parameters
     * @returns the outbound email record
     *
     * @example
     * ```typescript
     * const result = await email.send({
     *   from: 'support@myapp.agentuity.email',
     *   to: ['user@example.com'],
     *   subject: 'Welcome!',
     *   text: 'Welcome to our platform.',
     *   html: '<h1>Welcome!</h1><p>Welcome to our platform.</p>',
     * });
     * console.log('Sent:', result.id, 'Status:', result.status);
     * ```
     */
    send(params: EmailSendParams): Promise<EmailOutbound>;
    /**
     * List inbound emails
     *
     * @param addressId - optional email address ID to filter by
     * @returns array of inbound emails
     *
     * @example
     * ```typescript
     * const inbound = await email.listInbound('addr_123');
     * for (const msg of inbound) {
     *   console.log(`From: ${msg.from}, Subject: ${msg.subject}`);
     * }
     * ```
     */
    listInbound(addressId?: string): Promise<EmailInbound[]>;
    /**
     * Get an inbound email by ID
     *
     * @param id - the inbound email ID
     * @returns the inbound email or null if not found
     *
     * @example
     * ```typescript
     * const msg = await email.getInbound('inb_123');
     * if (msg) {
     *   console.log('Subject:', msg.subject);
     * }
     * ```
     */
    getInbound(id: string): Promise<EmailInbound | null>;
    /**
     * Delete an inbound email by ID
     *
     * @param id - the inbound email ID
     *
     * @example
     * ```typescript
     * await email.deleteInbound('inb_abc123');
     * ```
     */
    deleteInbound(id: string): Promise<void>;
    /**
     * List outbound emails
     *
     * @param addressId - optional email address ID to filter by
     * @returns array of outbound emails
     *
     * @example
     * ```typescript
     * const outbound = await email.listOutbound('addr_123');
     * for (const msg of outbound) {
     *   console.log(`To: ${msg.to}, Status: ${msg.status}`);
     * }
     * ```
     */
    listOutbound(addressId?: string): Promise<EmailOutbound[]>;
    /**
     * Get an outbound email by ID
     *
     * @param id - the outbound email ID
     * @returns the outbound email or null if not found
     *
     * @example
     * ```typescript
     * const msg = await email.getOutbound('out_123');
     * if (msg) {
     *   console.log('Status:', msg.status);
     * }
     * ```
     */
    getOutbound(id: string): Promise<EmailOutbound | null>;
    /**
     * Delete an outbound email by ID
     *
     * @param id - the outbound email ID
     *
     * @example
     * ```typescript
     * await email.deleteOutbound('out_abc123');
     * ```
     */
    deleteOutbound(id: string): Promise<void>;
    /**
     * Get email activity time-series data
     *
     * @param params - optional parameters (days defaults to 7)
     * @returns activity data points and the number of days queried
     *
     * @example
     * ```typescript
     * const activity = await email.getActivity({ days: 30 });
     * for (const point of activity.activity) {
     *   console.log(`${point.date}: ${point.inbound} in, ${point.outbound} out`);
     * }
     * ```
     */
    getActivity(params?: EmailActivityParams): Promise<EmailActivityResult>;
}
export declare class EmailStorageService implements EmailService {
    #private;
    constructor(baseUrl: string, adapter: FetchAdapter);
    createAddress(localPart: string): Promise<EmailAddress>;
    listAddresses(): Promise<EmailAddress[]>;
    getAddress(id: string): Promise<EmailAddress | null>;
    getConnectionConfig(id: string): Promise<EmailConnectionConfig | null>;
    deleteAddress(id: string): Promise<void>;
    createDestination(addressId: string, type: string, config: Record<string, unknown>): Promise<EmailDestination>;
    listDestinations(addressId: string): Promise<EmailDestination[]>;
    deleteDestination(addressId: string, destinationId: string): Promise<void>;
    send(params: EmailSendParams): Promise<EmailOutbound>;
    listInbound(addressId?: string): Promise<EmailInbound[]>;
    getInbound(id: string): Promise<EmailInbound | null>;
    deleteInbound(id: string): Promise<void>;
    listOutbound(addressId?: string): Promise<EmailOutbound[]>;
    getOutbound(id: string): Promise<EmailOutbound | null>;
    deleteOutbound(id: string): Promise<void>;
    getActivity(params?: EmailActivityParams): Promise<EmailActivityResult>;
}
//# sourceMappingURL=email.d.ts.map