import { success, createParseMethods, SCHEMA_KIND } from "../base.js";
import { optional } from "../utils/optional.js";
import { nullable } from "../utils/nullable.js";
const parseMethods = createParseMethods();
/**
 * Schema that coerces values to booleans using Boolean(value).
 * Uses JavaScript truthy/falsy rules.
 *
 * @example
 * ```typescript
 * const schema = s.coerce.boolean();
 * schema.parse(1); // true
 * schema.parse(0); // false
 * schema.parse(''); // false
 * schema.parse('hello'); // true
 * ```
 */
export class CoerceBooleanSchema {
    [SCHEMA_KIND] = 'CoerceBooleanSchema';
    description;
    '~standard' = {
        version: 1,
        vendor: 'agentuity',
        validate: (value) => {
            // Coerce to boolean using JavaScript truthiness rules
            return success(Boolean(value));
        },
        types: undefined,
    };
    describe(description) {
        this.description = description;
        return this;
    }
    optional() {
        return optional(this);
    }
    nullable() {
        return nullable(this);
    }
    parse = parseMethods.parse;
    safeParse = parseMethods.safeParse;
}
/**
 * Create a schema that coerces values to booleans.
 * Useful for parsing checkboxes or boolean flags from strings.
 */
export function coerceBoolean() {
    return new CoerceBooleanSchema();
}
//# sourceMappingURL=boolean.js.map