/**
 * Message handler callback type
 */
export type MessageHandler<T = unknown> = (data: T) => void;
/**
 * WebSocket state change callback types
 */
export interface WebSocketCallbacks<TOutput = unknown> {
    /** Called when connection is established */
    onConnect?: () => void;
    /** Called when connection is closed */
    onDisconnect?: () => void;
    /** Called when an error occurs */
    onError?: (error: Error) => void;
    /** Called when a message is received */
    onMessage?: MessageHandler<TOutput>;
}
/**
 * Options for WebSocketManager
 */
export interface WebSocketManagerOptions<TOutput = unknown> {
    /** WebSocket URL */
    url: string;
    /** Callbacks for state changes */
    callbacks?: WebSocketCallbacks<TOutput>;
    /** Reconnection configuration */
    reconnect?: {
        threshold?: number;
        baseDelay?: number;
        factor?: number;
        maxDelay?: number;
        jitter?: number;
    };
}
/**
 * WebSocket manager state
 */
export interface WebSocketManagerState {
    /** Whether WebSocket is currently connected */
    isConnected: boolean;
    /** Current error, if any */
    error: Error | null;
    /** WebSocket ready state */
    readyState: number;
}
/**
 * Generic WebSocket connection manager with automatic reconnection,
 * message queuing, and handler management.
 *
 * Framework-agnostic - can be used with React, Svelte, Vue, or vanilla JS.
 */
export declare class WebSocketManager<TInput = unknown, TOutput = unknown> {
    private ws;
    private manualClose;
    private pendingMessages;
    private queuedMessages;
    private messageHandler;
    private reconnectManager;
    private callbacks;
    private url;
    private reconnectConfig;
    constructor(options: WebSocketManagerOptions<TOutput>);
    /**
     * Connect to the WebSocket server
     */
    connect(): void;
    /**
     * Send data through the WebSocket.
     * Messages are queued if not currently connected.
     */
    send(data: TInput): void;
    /**
     * Set the message handler.
     * Any buffered messages will be delivered immediately.
     */
    setMessageHandler(handler: MessageHandler<TOutput>): void;
    /**
     * Get current state
     */
    getState(): WebSocketManagerState;
    /**
     * Close the WebSocket connection and cleanup
     */
    close(): void;
    /**
     * Dispose of the manager (alias for close)
     */
    dispose(): void;
}
//# sourceMappingURL=websocket-manager.d.ts.map