import { buildUrl, toServiceException } from "./_util.js";
import { StructuredError } from "../error.js";
import { safeStringify } from "../json.js";
/** API version string used for task CRUD, comment, tag, and attachment endpoints. */
const TASK_API_VERSION = '2026-02-24';
/** API version string used for the task activity analytics endpoint. */
const TASK_ACTIVITY_API_VERSION = '2026-02-28';
/** Thrown when a task ID parameter is empty or not a string. */
const TaskIdRequiredError = StructuredError('TaskIdRequiredError', 'Task ID is required and must be a non-empty string');
/** Thrown when a task title is empty or not a string. */
const TaskTitleRequiredError = StructuredError('TaskTitleRequiredError', 'Task title is required and must be a non-empty string');
/** Thrown when a comment ID parameter is empty or not a string. */
const CommentIdRequiredError = StructuredError('CommentIdRequiredError', 'Comment ID is required and must be a non-empty string');
/** Thrown when a comment body is empty or not a string. */
const CommentBodyRequiredError = StructuredError('CommentBodyRequiredError', 'Comment body is required and must be a non-empty string');
/** Thrown when a tag ID parameter is empty or not a string. */
const TagIdRequiredError = StructuredError('TagIdRequiredError', 'Tag ID is required and must be a non-empty string');
/** Thrown when a tag name is empty or not a string. */
const TagNameRequiredError = StructuredError('TagNameRequiredError', 'Tag name is required and must be a non-empty string');
/** Thrown when an attachment ID parameter is empty or not a string. */
const AttachmentIdRequiredError = StructuredError('AttachmentIdRequiredError', 'Attachment ID is required and must be a non-empty string');
/** Thrown when a user ID parameter is empty or not a string. */
const UserIdRequiredError = StructuredError('UserIdRequiredError', 'User ID is required and must be a non-empty string');
/**
 * Thrown when the API returns a success HTTP status but the response body indicates failure.
 */
const TaskStorageResponseError = StructuredError('TaskStorageResponseError')();
/**
 * Client for the Agentuity Task management service.
 *
 * Provides a full-featured project management API including task CRUD, hierarchical
 * organization (epics → features → tasks), comments, tags, file attachments via
 * presigned S3 URLs, changelog tracking, and activity analytics.
 *
 * Tasks support lifecycle management through status transitions (`open` → `in_progress`
 * → `done`/`closed`/`cancelled`) with automatic date tracking for each transition.
 *
 * All methods validate inputs client-side and throw structured errors for invalid
 * parameters. API errors throw {@link ServiceException}.
 *
 * @example
 * ```typescript
 * const tasks = new TaskStorageService(baseUrl, adapter);
 *
 * // Create a task
 * const task = await tasks.create({
 *   title: 'Implement login flow',
 *   type: 'feature',
 *   created_id: 'user_123',
 *   creator: { id: 'user_123', name: 'Alice' },
 *   priority: 'high',
 * });
 *
 * // Add a comment
 * await tasks.createComment(task.id, 'Started working on this', 'user_123');
 *
 * // List open tasks
 * const { tasks: openTasks } = await tasks.list({ status: 'open' });
 * ```
 */
export class TaskStorageService {
    #adapter;
    #baseUrl;
    /**
     * Creates a new TaskStorageService instance.
     *
     * @param baseUrl - The base URL of the task management API
     * @param adapter - The HTTP fetch adapter used for making API requests
     */
    constructor(baseUrl, adapter) {
        this.#adapter = adapter;
        this.#baseUrl = baseUrl;
    }
    /**
     * Create a new task.
     *
     * @param params - The task creation parameters including title, type, and optional fields
     * @returns The newly created task
     * @throws {@link TaskTitleRequiredError} if the title is empty or not a string
     * @throws {@link ServiceException} if the API request fails
     *
     * @example
     * ```typescript
     * const task = await tasks.create({
     *   title: 'Fix login bug',
     *   type: 'bug',
     *   created_id: 'user_123',
     *   priority: 'high',
     *   creator: { id: 'user_123', name: 'Alice' },
     *   project: { id: 'proj_456', name: 'Auth Service' },
     * });
     * console.log('Created:', task.id);
     * ```
     */
    async create(params) {
        if (!params?.title || typeof params.title !== 'string' || params.title.trim().length === 0) {
            throw new TaskTitleRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/${TASK_API_VERSION}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'POST',
            body: safeStringify(params),
            contentType: 'application/json',
            signal,
            telemetry: {
                name: 'agentuity.task.create',
                attributes: {
                    type: params.type,
                    priority: params.priority ?? 'none',
                    status: params.status ?? 'open',
                },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('POST', url, res.response);
    }
    /**
     * Get a task by its ID.
     *
     * @param id - The unique task identifier
     * @returns The task if found, or `null` if the task does not exist
     * @throws {@link TaskIdRequiredError} if the ID is empty or not a string
     * @throws {@link ServiceException} if the API request fails
     *
     * @example
     * ```typescript
     * const task = await tasks.get('task_abc123');
     * if (task) {
     *   console.log(task.title, task.status);
     * } else {
     *   console.log('Task not found');
     * }
     * ```
     */
    async get(id) {
        if (!id || typeof id !== 'string' || id.trim().length === 0) {
            throw new TaskIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/${TASK_API_VERSION}/${encodeURIComponent(id)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.task.get',
                attributes: { id },
            },
        });
        if (res.response.status === 404) {
            return null;
        }
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('GET', url, res.response);
    }
    /**
     * List tasks with optional filtering and pagination.
     *
     * @param params - Optional filter and pagination parameters
     * @returns Paginated list of tasks matching the filters
     * @throws {@link ServiceException} if the API request fails
     *
     * @example
     * ```typescript
     * // List all open high-priority bugs
     * const result = await tasks.list({
     *   status: 'open',
     *   type: 'bug',
     *   priority: 'high',
     *   sort: '-created_at',
     *   limit: 20,
     * });
     * console.log(`Found ${result.total} bugs, showing ${result.tasks.length}`);
     * ```
     */
    async list(params) {
        const queryParams = new URLSearchParams();
        if (params?.status)
            queryParams.set('status', params.status);
        if (params?.type)
            queryParams.set('type', params.type);
        if (params?.priority)
            queryParams.set('priority', params.priority);
        if (params?.assigned_id)
            queryParams.set('assigned_id', params.assigned_id);
        if (params?.parent_id)
            queryParams.set('parent_id', params.parent_id);
        if (params?.project_id)
            queryParams.set('project_id', params.project_id);
        if (params?.tag_id)
            queryParams.set('tag_id', params.tag_id);
        if (params?.deleted !== undefined)
            queryParams.set('deleted', String(params.deleted));
        if (params?.sort)
            queryParams.set('sort', params.sort);
        if (params?.order)
            queryParams.set('order', params.order);
        if (params?.limit !== undefined)
            queryParams.set('limit', String(params.limit));
        if (params?.offset !== undefined)
            queryParams.set('offset', String(params.offset));
        const queryString = queryParams.toString();
        const url = buildUrl(this.#baseUrl, `/task/${TASK_API_VERSION}${queryString ? `?${queryString}` : ''}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.task.list',
                attributes: {
                    ...(params?.status ? { status: params.status } : {}),
                    ...(params?.type ? { type: params.type } : {}),
                    ...(params?.priority ? { priority: params.priority } : {}),
                },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('GET', url, res.response);
    }
    /**
     * Partially update an existing task.
     *
     * @param id - The unique task identifier
     * @param params - Fields to update; only provided fields are changed
     * @returns The updated task
     * @throws {@link TaskIdRequiredError} if the ID is empty or not a string
     * @throws {@link TaskTitleRequiredError} if a title is provided but is empty
     * @throws {@link ServiceException} if the API request fails
     *
     * @example
     * ```typescript
     * const updated = await tasks.update('task_abc123', {
     *   status: 'in_progress',
     *   priority: 'high',
     *   assignee: { id: 'user_456', name: 'Bob' },
     * });
     * console.log('Updated status:', updated.status);
     * ```
     */
    async update(id, params) {
        if (!id || typeof id !== 'string' || id.trim().length === 0) {
            throw new TaskIdRequiredError();
        }
        if (params.title !== undefined &&
            (typeof params.title !== 'string' || params.title.trim().length === 0)) {
            throw new TaskTitleRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/${TASK_API_VERSION}/${encodeURIComponent(id)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'PATCH',
            body: safeStringify(params),
            contentType: 'application/json',
            signal,
            telemetry: {
                name: 'agentuity.task.update',
                attributes: { id },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('PATCH', url, res.response);
    }
    /**
     * Close a task by setting its status to closed.
     *
     * @param id - The unique task identifier
     * @returns The closed task with updated `closed_date`
     * @throws {@link TaskIdRequiredError} if the ID is empty or not a string
     * @throws {@link ServiceException} if the API request fails
     *
     * @example
     * ```typescript
     * const closed = await tasks.close('task_abc123');
     * console.log('Closed at:', closed.closed_date);
     * ```
     */
    async close(id) {
        if (!id || typeof id !== 'string' || id.trim().length === 0) {
            throw new TaskIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/${TASK_API_VERSION}/${encodeURIComponent(id)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'DELETE',
            signal,
            telemetry: {
                name: 'agentuity.task.close',
                attributes: { id },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('DELETE', url, res.response);
    }
    /**
     * Get the changelog (audit trail) for a task, showing all field changes over time.
     *
     * @param id - The unique task identifier
     * @param params - Optional pagination parameters
     * @returns Paginated list of changelog entries ordered by most recent first
     * @throws {@link TaskIdRequiredError} if the ID is empty or not a string
     * @throws {@link ServiceException} if the API request fails
     *
     * @example
     * ```typescript
     * const { changelog, total } = await tasks.changelog('task_abc123', {
     *   limit: 10,
     *   offset: 0,
     * });
     * for (const entry of changelog) {
     *   console.log(`${entry.field}: ${entry.old_value} → ${entry.new_value}`);
     * }
     * ```
     */
    async changelog(id, params) {
        if (!id || typeof id !== 'string' || id.trim().length === 0) {
            throw new TaskIdRequiredError();
        }
        const queryParams = new URLSearchParams();
        if (params?.limit !== undefined)
            queryParams.set('limit', String(params.limit));
        if (params?.offset !== undefined)
            queryParams.set('offset', String(params.offset));
        const queryString = queryParams.toString();
        const url = buildUrl(this.#baseUrl, `/task/changelog/${TASK_API_VERSION}/${encodeURIComponent(id)}${queryString ? `?${queryString}` : ''}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.task.changelog',
                attributes: { id },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('GET', url, res.response);
    }
    /**
     * Soft-delete a task, marking it as deleted without permanent removal.
     *
     * @param id - The unique task identifier
     * @returns The soft-deleted task
     * @throws {@link TaskIdRequiredError} if the ID is empty or not a string
     * @throws {@link ServiceException} if the API request fails
     *
     * @example
     * ```typescript
     * const deleted = await tasks.softDelete('task_abc123');
     * console.log('Soft-deleted task:', deleted.id);
     * ```
     */
    async softDelete(id) {
        if (!id || typeof id !== 'string' || id.trim().length === 0) {
            throw new TaskIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/delete/${TASK_API_VERSION}/${encodeURIComponent(id)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'POST',
            signal,
            telemetry: {
                name: 'agentuity.task.softDelete',
                attributes: { id },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('POST', url, res.response);
    }
    /**
     * Create a comment on a task.
     *
     * @param taskId - The ID of the task to comment on
     * @param body - The comment text content (must be non-empty)
     * @param userId - The ID of the user authoring the comment
     * @param author - Optional entity reference with the author's display name
     * @returns The newly created comment
     * @throws {@link TaskIdRequiredError} if the task ID is empty or not a string
     * @throws {@link CommentBodyRequiredError} if the body is empty or not a string
     * @throws {@link UserIdRequiredError} if the user ID is empty or not a string
     * @throws {@link ServiceException} if the API request fails
     *
     * @example
     * ```typescript
     * const comment = await tasks.createComment(
     *   'task_abc123',
     *   'This is ready for review.',
     *   'user_456',
     *   { id: 'user_456', name: 'Bob' },
     * );
     * console.log('Comment created:', comment.id);
     * ```
     */
    async createComment(taskId, body, userId, author) {
        if (!taskId || typeof taskId !== 'string' || taskId.trim().length === 0) {
            throw new TaskIdRequiredError();
        }
        if (!body || typeof body !== 'string' || body.trim().length === 0) {
            throw new CommentBodyRequiredError();
        }
        if (!userId || typeof userId !== 'string' || userId.trim().length === 0) {
            throw new UserIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/comments/create/${TASK_API_VERSION}/${encodeURIComponent(taskId)}`);
        const signal = AbortSignal.timeout(30_000);
        const commentBody = { body, user_id: userId };
        if (author)
            commentBody.author = author;
        const res = await this.#adapter.invoke(url, {
            method: 'POST',
            body: safeStringify(commentBody),
            contentType: 'application/json',
            signal,
            telemetry: {
                name: 'agentuity.task.createComment',
                attributes: { taskId },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('POST', url, res.response);
    }
    /**
     * Get a comment by its ID.
     *
     * @param commentId - The unique comment identifier
     * @returns The comment
     * @throws {@link CommentIdRequiredError} if the comment ID is empty or not a string
     * @throws {@link ServiceException} if the API request fails
     *
     * @example
     * ```typescript
     * const comment = await tasks.getComment('comment_xyz789');
     * console.log(`${comment.author?.name}: ${comment.body}`);
     * ```
     */
    async getComment(commentId) {
        if (!commentId || typeof commentId !== 'string' || commentId.trim().length === 0) {
            throw new CommentIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/comments/get/${TASK_API_VERSION}/${encodeURIComponent(commentId)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.task.getComment',
                attributes: { commentId },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('GET', url, res.response);
    }
    /**
     * Update a comment's body text.
     *
     * @param commentId - The unique comment identifier
     * @param body - The new comment text (must be non-empty)
     * @returns The updated comment
     * @throws {@link CommentIdRequiredError} if the comment ID is empty or not a string
     * @throws {@link CommentBodyRequiredError} if the body is empty or not a string
     * @throws {@link ServiceException} if the API request fails
     *
     * @example
     * ```typescript
     * const updated = await tasks.updateComment(
     *   'comment_xyz789',
     *   'Updated: This is now ready for final review.',
     * );
     * console.log('Updated at:', updated.updated_at);
     * ```
     */
    async updateComment(commentId, body) {
        if (!commentId || typeof commentId !== 'string' || commentId.trim().length === 0) {
            throw new CommentIdRequiredError();
        }
        if (!body || typeof body !== 'string' || body.trim().length === 0) {
            throw new CommentBodyRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/comments/update/${TASK_API_VERSION}/${encodeURIComponent(commentId)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'PATCH',
            body: safeStringify({ body }),
            contentType: 'application/json',
            signal,
            telemetry: {
                name: 'agentuity.task.updateComment',
                attributes: { commentId },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('PATCH', url, res.response);
    }
    /**
     * Delete a comment permanently.
     *
     * @param commentId - The unique comment identifier
     * @throws {@link CommentIdRequiredError} if the comment ID is empty or not a string
     * @throws {@link ServiceException} if the API request fails
     *
     * @example
     * ```typescript
     * await tasks.deleteComment('comment_xyz789');
     * console.log('Comment deleted');
     * ```
     */
    async deleteComment(commentId) {
        if (!commentId || typeof commentId !== 'string' || commentId.trim().length === 0) {
            throw new CommentIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/comments/delete/${TASK_API_VERSION}/${encodeURIComponent(commentId)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'DELETE',
            signal,
            telemetry: {
                name: 'agentuity.task.deleteComment',
                attributes: { commentId },
            },
        });
        if (res.ok) {
            if (res.data?.success === false) {
                throw new TaskStorageResponseError({
                    status: res.response.status,
                    message: res.data.message ?? 'Operation failed',
                });
            }
            return;
        }
        throw await toServiceException('DELETE', url, res.response);
    }
    /**
     * List comments on a task with optional pagination.
     *
     * @param taskId - The ID of the task whose comments to list
     * @param params - Optional pagination parameters
     * @returns Paginated list of comments
     * @throws {@link TaskIdRequiredError} if the task ID is empty or not a string
     * @throws {@link ServiceException} if the API request fails
     *
     * @example
     * ```typescript
     * const { comments, total } = await tasks.listComments('task_abc123', {
     *   limit: 25,
     *   offset: 0,
     * });
     * for (const c of comments) {
     *   console.log(`${c.author?.name}: ${c.body}`);
     * }
     * ```
     */
    async listComments(taskId, params) {
        if (!taskId || typeof taskId !== 'string' || taskId.trim().length === 0) {
            throw new TaskIdRequiredError();
        }
        const queryParams = new URLSearchParams();
        if (params?.limit !== undefined)
            queryParams.set('limit', String(params.limit));
        if (params?.offset !== undefined)
            queryParams.set('offset', String(params.offset));
        const queryString = queryParams.toString();
        const url = buildUrl(this.#baseUrl, `/task/comments/list/${TASK_API_VERSION}/${encodeURIComponent(taskId)}${queryString ? `?${queryString}` : ''}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.task.listComments',
                attributes: { taskId },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('GET', url, res.response);
    }
    /**
     * Create a new tag for categorizing tasks.
     *
     * @param name - The tag display name (must be non-empty)
     * @param color - Optional hex color code (e.g., `'#ff0000'`)
     * @returns The newly created tag
     * @throws {@link TagNameRequiredError} if the name is empty or not a string
     * @throws {@link ServiceException} if the API request fails
     *
     * @example
     * ```typescript
     * const tag = await tasks.createTag('urgent', '#ff0000');
     * console.log('Created tag:', tag.id, tag.name);
     * ```
     */
    async createTag(name, color) {
        if (!name || typeof name !== 'string' || name.trim().length === 0) {
            throw new TagNameRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/tags/create/${TASK_API_VERSION}`);
        const signal = AbortSignal.timeout(30_000);
        const body = { name };
        if (color !== undefined)
            body.color = color;
        const res = await this.#adapter.invoke(url, {
            method: 'POST',
            body: safeStringify(body),
            contentType: 'application/json',
            signal,
            telemetry: {
                name: 'agentuity.task.createTag',
                attributes: { tagName: name },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('POST', url, res.response);
    }
    /**
     * Get a tag by its ID.
     *
     * @param tagId - The unique tag identifier
     * @returns The tag
     * @throws {@link TagIdRequiredError} if the tag ID is empty or not a string
     * @throws {@link ServiceException} if the API request fails
     *
     * @example
     * ```typescript
     * const tag = await tasks.getTag('tag_def456');
     * console.log(`${tag.name} (${tag.color})`);
     * ```
     */
    async getTag(tagId) {
        if (!tagId || typeof tagId !== 'string' || tagId.trim().length === 0) {
            throw new TagIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/tags/get/${TASK_API_VERSION}/${encodeURIComponent(tagId)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.task.getTag',
                attributes: { tagId },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('GET', url, res.response);
    }
    /**
     * Update a tag's name and optionally its color.
     *
     * @param tagId - The unique tag identifier
     * @param name - The new tag name (must be non-empty)
     * @param color - Optional new hex color code
     * @returns The updated tag
     * @throws {@link TagIdRequiredError} if the tag ID is empty or not a string
     * @throws {@link TagNameRequiredError} if the name is empty or not a string
     * @throws {@link ServiceException} if the API request fails
     *
     * @example
     * ```typescript
     * const updated = await tasks.updateTag('tag_def456', 'critical', '#cc0000');
     * console.log('Updated:', updated.name);
     * ```
     */
    async updateTag(tagId, name, color) {
        if (!tagId || typeof tagId !== 'string' || tagId.trim().length === 0) {
            throw new TagIdRequiredError();
        }
        if (!name || typeof name !== 'string' || name.trim().length === 0) {
            throw new TagNameRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/tags/update/${TASK_API_VERSION}/${encodeURIComponent(tagId)}`);
        const signal = AbortSignal.timeout(30_000);
        const body = { name };
        if (color !== undefined)
            body.color = color;
        const res = await this.#adapter.invoke(url, {
            method: 'PATCH',
            body: safeStringify(body),
            contentType: 'application/json',
            signal,
            telemetry: {
                name: 'agentuity.task.updateTag',
                attributes: { tagId },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('PATCH', url, res.response);
    }
    /**
     * Delete a tag permanently.
     *
     * @param tagId - The unique tag identifier
     * @throws {@link TagIdRequiredError} if the tag ID is empty or not a string
     * @throws {@link ServiceException} if the API request fails
     *
     * @example
     * ```typescript
     * await tasks.deleteTag('tag_def456');
     * console.log('Tag deleted');
     * ```
     */
    async deleteTag(tagId) {
        if (!tagId || typeof tagId !== 'string' || tagId.trim().length === 0) {
            throw new TagIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/tags/delete/${TASK_API_VERSION}/${encodeURIComponent(tagId)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'DELETE',
            signal,
            telemetry: {
                name: 'agentuity.task.deleteTag',
                attributes: { tagId },
            },
        });
        if (res.ok) {
            if (res.data?.success === false) {
                throw new TaskStorageResponseError({
                    status: res.response.status,
                    message: res.data.message ?? 'Operation failed',
                });
            }
            return;
        }
        throw await toServiceException('DELETE', url, res.response);
    }
    /**
     * List all tags in the organization.
     *
     * @returns List of all tags
     * @throws {@link ServiceException} if the API request fails
     *
     * @example
     * ```typescript
     * const { tags } = await tasks.listTags();
     * for (const tag of tags) {
     *   console.log(`${tag.name} (${tag.color ?? 'no color'})`);
     * }
     * ```
     */
    async listTags() {
        const url = buildUrl(this.#baseUrl, `/task/tags/list/${TASK_API_VERSION}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.task.listTags',
                attributes: {},
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('GET', url, res.response);
    }
    /**
     * Associate a tag with a task.
     *
     * @param taskId - The ID of the task
     * @param tagId - The ID of the tag to add
     * @throws {@link TaskIdRequiredError} if the task ID is empty or not a string
     * @throws {@link TagIdRequiredError} if the tag ID is empty or not a string
     * @throws {@link ServiceException} if the API request fails
     *
     * @example
     * ```typescript
     * await tasks.addTagToTask('task_abc123', 'tag_def456');
     * console.log('Tag added to task');
     * ```
     */
    async addTagToTask(taskId, tagId) {
        if (!taskId || typeof taskId !== 'string' || taskId.trim().length === 0) {
            throw new TaskIdRequiredError();
        }
        if (!tagId || typeof tagId !== 'string' || tagId.trim().length === 0) {
            throw new TagIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/tags/add/${TASK_API_VERSION}/${encodeURIComponent(taskId)}/${encodeURIComponent(tagId)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'POST',
            signal,
            telemetry: {
                name: 'agentuity.task.addTagToTask',
                attributes: { taskId, tagId },
            },
        });
        if (res.ok) {
            if (res.data?.success === false) {
                throw new TaskStorageResponseError({
                    status: res.response.status,
                    message: res.data.message ?? 'Operation failed',
                });
            }
            return;
        }
        throw await toServiceException('POST', url, res.response);
    }
    /**
     * Remove a tag association from a task.
     *
     * @param taskId - The ID of the task
     * @param tagId - The ID of the tag to remove
     * @throws {@link TaskIdRequiredError} if the task ID is empty or not a string
     * @throws {@link TagIdRequiredError} if the tag ID is empty or not a string
     * @throws {@link ServiceException} if the API request fails
     *
     * @example
     * ```typescript
     * await tasks.removeTagFromTask('task_abc123', 'tag_def456');
     * console.log('Tag removed from task');
     * ```
     */
    async removeTagFromTask(taskId, tagId) {
        if (!taskId || typeof taskId !== 'string' || taskId.trim().length === 0) {
            throw new TaskIdRequiredError();
        }
        if (!tagId || typeof tagId !== 'string' || tagId.trim().length === 0) {
            throw new TagIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/tags/remove/${TASK_API_VERSION}/${encodeURIComponent(taskId)}/${encodeURIComponent(tagId)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'DELETE',
            signal,
            telemetry: {
                name: 'agentuity.task.removeTagFromTask',
                attributes: { taskId, tagId },
            },
        });
        if (res.ok) {
            if (res.data?.success === false) {
                throw new TaskStorageResponseError({
                    status: res.response.status,
                    message: res.data.message ?? 'Operation failed',
                });
            }
            return;
        }
        throw await toServiceException('DELETE', url, res.response);
    }
    /**
     * List all tags associated with a specific task.
     *
     * @param taskId - The ID of the task
     * @returns Array of tags on the task
     * @throws {@link TaskIdRequiredError} if the task ID is empty or not a string
     * @throws {@link ServiceException} if the API request fails
     *
     * @example
     * ```typescript
     * const tags = await tasks.listTagsForTask('task_abc123');
     * console.log('Tags:', tags.map((t) => t.name).join(', '));
     * ```
     */
    async listTagsForTask(taskId) {
        if (!taskId || typeof taskId !== 'string' || taskId.trim().length === 0) {
            throw new TaskIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/tags/task/${TASK_API_VERSION}/${encodeURIComponent(taskId)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.task.listTagsForTask',
                attributes: { taskId },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('GET', url, res.response);
    }
    /**
     * Initiate a file upload to a task. Returns a presigned S3 URL for direct upload.
     *
     * @remarks
     * After receiving the presigned URL, upload the file content via HTTP PUT to that URL.
     * Then call {@link TaskStorageService.confirmAttachment | confirmAttachment} to finalize.
     *
     * @param taskId - The ID of the task to attach the file to
     * @param params - Attachment metadata including filename, content type, and size
     * @returns The created attachment record and a presigned upload URL
     * @throws {@link TaskIdRequiredError} if the task ID is empty or not a string
     * @throws {@link ServiceException} if the API request fails
     *
     * @example
     * ```typescript
     * const { attachment, presigned_url } = await tasks.uploadAttachment(
     *   'task_abc123',
     *   { filename: 'report.pdf', content_type: 'application/pdf', size: 102400 },
     * );
     *
     * // Upload the file to S3
     * await fetch(presigned_url, { method: 'PUT', body: fileContent });
     *
     * // Confirm the upload
     * await tasks.confirmAttachment(attachment.id);
     * ```
     */
    async uploadAttachment(taskId, params) {
        if (!taskId || typeof taskId !== 'string' || taskId.trim().length === 0) {
            throw new TaskIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/attachments/presign-upload/${TASK_API_VERSION}/${encodeURIComponent(taskId)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'POST',
            body: safeStringify(params),
            contentType: 'application/json',
            signal,
            telemetry: {
                name: 'agentuity.task.uploadAttachment',
                attributes: { taskId },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('POST', url, res.response);
    }
    /**
     * Confirm that a file upload has completed successfully.
     *
     * @remarks
     * Call this after successfully uploading the file to the presigned URL
     * returned by {@link TaskStorageService.uploadAttachment | uploadAttachment}.
     *
     * @param attachmentId - The unique attachment identifier
     * @returns The confirmed attachment record
     * @throws {@link AttachmentIdRequiredError} if the attachment ID is empty or not a string
     * @throws {@link ServiceException} if the API request fails
     *
     * @example
     * ```typescript
     * const confirmed = await tasks.confirmAttachment('att_ghi789');
     * console.log('Confirmed:', confirmed.filename);
     * ```
     */
    async confirmAttachment(attachmentId) {
        if (!attachmentId || typeof attachmentId !== 'string' || attachmentId.trim().length === 0) {
            throw new AttachmentIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/attachments/confirm/${TASK_API_VERSION}/${encodeURIComponent(attachmentId)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'POST',
            signal,
            telemetry: {
                name: 'agentuity.task.confirmAttachment',
                attributes: { attachmentId },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('POST', url, res.response);
    }
    /**
     * Get a presigned S3 URL for downloading an attachment.
     *
     * @param attachmentId - The unique attachment identifier
     * @returns A presigned download URL with expiry information
     * @throws {@link AttachmentIdRequiredError} if the attachment ID is empty or not a string
     * @throws {@link ServiceException} if the API request fails
     *
     * @example
     * ```typescript
     * const { presigned_url, expiry_seconds } = await tasks.downloadAttachment('att_ghi789');
     * console.log(`Download URL (expires in ${expiry_seconds}s):`, presigned_url);
     * ```
     */
    async downloadAttachment(attachmentId) {
        if (!attachmentId || typeof attachmentId !== 'string' || attachmentId.trim().length === 0) {
            throw new AttachmentIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/attachments/presign-download/${TASK_API_VERSION}/${encodeURIComponent(attachmentId)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'POST',
            signal,
            telemetry: {
                name: 'agentuity.task.downloadAttachment',
                attributes: { attachmentId },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('POST', url, res.response);
    }
    /**
     * List all attachments on a task.
     *
     * @param taskId - The ID of the task
     * @returns List of attachments with total count
     * @throws {@link TaskIdRequiredError} if the task ID is empty or not a string
     * @throws {@link ServiceException} if the API request fails
     *
     * @example
     * ```typescript
     * const { attachments, total } = await tasks.listAttachments('task_abc123');
     * for (const att of attachments) {
     *   console.log(`${att.filename} (${att.content_type}, ${att.size} bytes)`);
     * }
     * ```
     */
    async listAttachments(taskId) {
        if (!taskId || typeof taskId !== 'string' || taskId.trim().length === 0) {
            throw new TaskIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/attachments/list/${TASK_API_VERSION}/${encodeURIComponent(taskId)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.task.listAttachments',
                attributes: { taskId },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('GET', url, res.response);
    }
    /**
     * Delete an attachment permanently.
     *
     * @param attachmentId - The unique attachment identifier
     * @throws {@link AttachmentIdRequiredError} if the attachment ID is empty or not a string
     * @throws {@link ServiceException} if the API request fails
     *
     * @example
     * ```typescript
     * await tasks.deleteAttachment('att_ghi789');
     * console.log('Attachment deleted');
     * ```
     */
    async deleteAttachment(attachmentId) {
        if (!attachmentId || typeof attachmentId !== 'string' || attachmentId.trim().length === 0) {
            throw new AttachmentIdRequiredError();
        }
        const url = buildUrl(this.#baseUrl, `/task/attachments/delete/${TASK_API_VERSION}/${encodeURIComponent(attachmentId)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'DELETE',
            signal,
            telemetry: {
                name: 'agentuity.task.deleteAttachment',
                attributes: { attachmentId },
            },
        });
        if (res.ok) {
            if (res.data?.success === false) {
                throw new TaskStorageResponseError({
                    status: res.response.status,
                    message: res.data.message ?? 'Operation failed',
                });
            }
            return;
        }
        throw await toServiceException('DELETE', url, res.response);
    }
    /**
     * List all users who have been referenced in tasks (as creators, assignees, or closers).
     *
     * @returns List of user entity references
     * @throws {@link ServiceException} if the API request fails
     *
     * @example
     * ```typescript
     * const { users } = await tasks.listUsers();
     * for (const user of users) {
     *   console.log(`${user.name} (${user.id})`);
     * }
     * ```
     */
    async listUsers() {
        const url = buildUrl(this.#baseUrl, `/task/users/${TASK_API_VERSION}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.task.listUsers',
                attributes: {},
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('GET', url, res.response);
    }
    /**
     * List all projects that have been referenced in tasks.
     *
     * @returns List of project entity references
     * @throws {@link ServiceException} if the API request fails
     *
     * @example
     * ```typescript
     * const { projects } = await tasks.listProjects();
     * for (const project of projects) {
     *   console.log(`${project.name} (${project.id})`);
     * }
     * ```
     */
    async listProjects() {
        const url = buildUrl(this.#baseUrl, `/task/projects/${TASK_API_VERSION}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.task.listProjects',
                attributes: {},
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('GET', url, res.response);
    }
    /**
     * Get task activity time-series data showing daily task counts by status.
     *
     * @param params - Optional parameters controlling the number of days to retrieve
     * @returns Time-series activity data with daily snapshots
     * @throws {@link ServiceException} if the API request fails
     *
     * @example
     * ```typescript
     * const { activity, days } = await tasks.getActivity({ days: 30 });
     * console.log(`Activity over ${days} days:`);
     * for (const point of activity) {
     *   console.log(`${point.date}: ${point.open} open, ${point.inProgress} in progress`);
     * }
     * ```
     */
    async getActivity(params) {
        const queryParams = new URLSearchParams();
        if (params?.days !== undefined)
            queryParams.set('days', String(params.days));
        const queryString = queryParams.toString();
        const url = buildUrl(this.#baseUrl, `/task/activity/${TASK_ACTIVITY_API_VERSION}${queryString ? `?${queryString}` : ''}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.task.activity',
                attributes: {
                    ...(params?.days !== undefined ? { days: String(params.days) } : {}),
                },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new TaskStorageResponseError({
                status: res.response.status,
                message: res.data.message,
            });
        }
        throw await toServiceException('GET', url, res.response);
    }
}
//# sourceMappingURL=task.js.map