import { FetchAdapter } from './adapter.ts';
/**
 * An email address registered with the Agentuity email service.
 *
 * Email addresses are created under the `@agentuity.email` domain and can receive
 * inbound emails (forwarded to configured destinations) and send outbound emails.
 */
export interface EmailAddress {
    /**
     * Unique identifier for the email address.
     *
     * @remarks Prefixed with `eaddr_`.
     */
    id: string;
    /**
     * The full email address (e.g., `support@agentuity.email`).
     */
    email: string;
    /**
     * Provider-specific configuration (e.g., inbound routing config).
     *
     * @remarks Opaque to callers — the structure is managed by the platform.
     */
    config?: Record<string, unknown>;
    /**
     * ID of the user who registered this address.
     */
    created_by?: string;
    /**
     * ISO 8601 timestamp when the address was created.
     */
    created_at: string;
    /**
     * ISO 8601 timestamp when the address was last updated.
     */
    updated_at?: string;
    /**
     * Total number of inbound emails received at this address.
     */
    inbound_count?: number;
    /**
     * Total number of outbound emails sent from this address.
     */
    outbound_count?: number;
    /**
     * ISO 8601 timestamp of the most recent inbound or outbound email activity.
     */
    last_activity?: string;
}
/**
 * A destination configuration for an email address.
 *
 * When an inbound email is received at the parent address, the platform forwards
 * it to each configured destination via an HTTP request.
 */
export interface EmailDestination {
    /**
     * Unique identifier for the destination.
     *
     * @remarks Prefixed with `edst_`.
     */
    id: string;
    /**
     * The destination type. Currently only `'url'` is supported.
     */
    type: string;
    /**
     * Destination-specific configuration.
     *
     * @remarks
     * For `'url'` type the shape is:
     * ```typescript
     * {
     *   url: string;             // Must use http or https; must not point to private/loopback addresses
     *   headers?: Record<string, string>;
     *   method?: 'POST' | 'PUT' | 'PATCH';
     * }
     * ```
     */
    config?: Record<string, unknown>;
    /**
     * ISO 8601 timestamp when the destination was created.
     */
    created_at: string;
    /**
     * ISO 8601 timestamp when the destination was last updated.
     */
    updated_at?: string;
}
/**
 * Connection settings for an email protocol (IMAP or POP3).
 *
 * Used to configure a mail client for accessing an Agentuity email address
 * via standard mail protocols.
 */
export interface EmailProtocolConfig {
    /**
     * The mail server hostname.
     */
    host: string;
    /**
     * The mail server port number.
     */
    port: number;
    /**
     * TLS mode (e.g., `'starttls'`, `'ssl'`, `'none'`).
     */
    tls: string;
    /**
     * The authentication username (typically the address ID).
     */
    username: string;
    /**
     * The authentication password.
     */
    password: string;
}
/**
 * Full connection configuration for accessing an email address via IMAP and POP3 protocols.
 *
 * Returned by {@link EmailService.getConnectionConfig} to allow external mail clients
 * to connect to an Agentuity email address.
 */
export interface EmailConnectionConfig {
    /**
     * The full email address these settings are for.
     */
    email: string;
    /**
     * IMAP protocol connection settings.
     */
    imap: EmailProtocolConfig;
    /**
     * POP3 protocol connection settings.
     */
    pop3: EmailProtocolConfig;
}
/**
 * An inbound email message received at an Agentuity email address.
 */
export interface EmailInbound {
    /**
     * Unique identifier for the inbound email.
     *
     * @remarks Prefixed with `einb_`.
     */
    id: string;
    /**
     * The sender's email address.
     */
    from: string;
    /**
     * The recipient email address (comma-separated if multiple).
     */
    to: string;
    /**
     * The email subject line.
     */
    subject?: string;
    /**
     * Plain text body of the email.
     */
    text?: string;
    /**
     * HTML body of the email.
     */
    html?: string;
    /**
     * ISO 8601 timestamp when the email was received.
     */
    received_at?: string;
    /**
     * Raw email headers as key-value pairs.
     */
    headers?: Record<string, unknown>;
    /**
     * Array of stored attachment metadata with S3 locations.
     */
    attachments?: EmailStoredAttachment[];
}
/**
 * An outbound email message sent from an Agentuity email address.
 */
export interface EmailOutbound {
    /**
     * Unique identifier for the outbound email.
     *
     * @remarks Prefixed with `eout_`.
     */
    id: string;
    /**
     * The sender's email address (must be owned by the organization).
     */
    from: string;
    /**
     * The recipient email addresses (comma-separated).
     */
    to: string;
    /**
     * The email subject line.
     */
    subject?: string;
    /**
     * Plain text body of the email.
     */
    text?: string;
    /**
     * HTML body of the email.
     */
    html?: string;
    /**
     * Delivery status: `'pending'`, `'success'`, or `'failed'`.
     *
     * @remarks Emails are sent asynchronously, so the initial status is always `'pending'`.
     */
    status?: string;
    /**
     * Error message if the delivery failed.
     */
    error?: string;
    /**
     * ISO 8601 timestamp when the send was initiated.
     */
    created_at?: string;
    /**
     * Custom email headers that were included.
     */
    headers?: Record<string, unknown>;
    /**
     * Array of stored attachment metadata with S3 locations.
     */
    attachments?: EmailStoredAttachment[];
}
/**
 * An email attachment to include when sending an outbound email.
 */
export interface EmailAttachment {
    /**
     * The filename for the attachment
     */
    filename: string;
    /**
     * The base64-encoded content of the attachment
     */
    content: string;
    /**
     * The MIME content type of the attachment
     */
    contentType?: string;
}
/**
 * A stored email attachment with S3 location metadata.
 * Returned by inbound/outbound email queries — different from EmailAttachment used for sending.
 */
export interface EmailStoredAttachment {
    /** The original filename */
    filename: string;
    /** The MIME content type */
    content_type?: string;
    /** File size in bytes */
    size: number;
    /** The S3 bucket name where the attachment is stored */
    bucket: string;
    /** The S3 object key */
    key: string;
    /** Optional pre-signed download URL */
    url?: string;
}
/**
 * Parameters for sending an email
 */
export interface EmailSendParams {
    /**
     * The sender email address (must be owned by the organization)
     */
    from: string;
    /**
     * The recipient email addresses
     */
    to: string[];
    /**
     * The email subject
     */
    subject: string;
    /**
     * Plain text email body
     */
    text?: string;
    /**
     * HTML email body
     */
    html?: string;
    /**
     * File attachments
     */
    attachments?: EmailAttachment[];
    /**
     * Custom email headers (e.g., In-Reply-To, References for threading)
     */
    headers?: Record<string, string>;
}
/**
 * Parameters for querying email activity time-series data.
 */
export interface EmailActivityParams {
    /**
     * Number of days of activity to retrieve.
     *
     * @remarks Values below 7 are clamped to 7; values above 365 are clamped to 365.
     *
     * @default 7
     */
    days?: number;
}
/**
 * A single data point in the email activity time-series.
 */
export interface EmailActivityDataPoint {
    /**
     * The date in `YYYY-MM-DD` format.
     */
    date: string;
    /**
     * Number of inbound emails received on this date.
     */
    inbound: number;
    /**
     * Number of outbound emails sent on this date.
     */
    outbound: number;
}
/**
 * Result of an email activity query containing daily time-series data.
 */
export interface EmailActivityResult {
    /**
     * Array of daily activity data points, ordered chronologically.
     */
    activity: EmailActivityDataPoint[];
    /**
     * The number of days of data returned.
     */
    days: number;
}
/**
 * Email service for managing email addresses, destinations, and sending/receiving emails
 */
export interface EmailService {
    /**
     * Create a new email address
     *
     * @param localPart - the local part of the email address (before the @)
     * @returns the created email address
     *
     * @example
     * ```typescript
     * const address = await email.createAddress('support');
     * console.log('Created:', address.email);
     * ```
     */
    createAddress(localPart: string): Promise<EmailAddress>;
    /**
     * List all email addresses
     *
     * @returns array of email addresses
     *
     * @example
     * ```typescript
     * const addresses = await email.listAddresses();
     * for (const addr of addresses) {
     *   console.log(addr.email);
     * }
     * ```
     */
    listAddresses(): Promise<EmailAddress[]>;
    /**
     * Get an email address by ID
     *
     * @param id - the email address ID
     * @returns the email address or null if not found
     *
     * @example
     * ```typescript
     * const address = await email.getAddress('addr_123');
     * if (address) {
     *   console.log('Found:', address.email);
     * }
     * ```
     */
    getAddress(id: string): Promise<EmailAddress | null>;
    /**
     * Get email connection settings (IMAP/POP3) for an address
     *
     * @param id - the email address ID
     * @returns the connection configuration or null if not found
     */
    getConnectionConfig(id: string): Promise<EmailConnectionConfig | null>;
    /**
     * Delete an email address
     *
     * @param id - the email address ID
     *
     * @example
     * ```typescript
     * await email.deleteAddress('addr_123');
     * ```
     */
    deleteAddress(id: string): Promise<void>;
    /**
     * Create a destination for an email address
     *
     * @param addressId - the email address ID
     * @param type - the destination type (e.g., 'url', 'agent')
     * @param config - the destination configuration
     * @returns the created destination
     *
     * @example
     * ```typescript
     * const dest = await email.createDestination('addr_123', 'url', {
     *   url: 'https://example.com/webhook',
     * });
     * console.log('Created destination:', dest.id);
     * ```
     */
    createDestination(addressId: string, type: string, config: Record<string, unknown>): Promise<EmailDestination>;
    /**
     * List destinations for an email address
     *
     * @param addressId - the email address ID
     * @returns array of destinations
     *
     * @example
     * ```typescript
     * const destinations = await email.listDestinations('addr_123');
     * for (const dest of destinations) {
     *   console.log(`${dest.type}: ${dest.id}`);
     * }
     * ```
     */
    listDestinations(addressId: string): Promise<EmailDestination[]>;
    /**
     * Delete a destination from an email address
     *
     * @param addressId - the email address ID
     * @param destinationId - the destination ID
     *
     * @example
     * ```typescript
     * await email.deleteDestination('addr_123', 'dest_456');
     * ```
     */
    deleteDestination(addressId: string, destinationId: string): Promise<void>;
    /**
     * Send an email
     *
     * @param params - the send parameters
     * @returns the outbound email record
     *
     * @example
     * ```typescript
     * const result = await email.send({
     *   from: 'support@myapp.agentuity.email',
     *   to: ['user@example.com'],
     *   subject: 'Welcome!',
     *   text: 'Welcome to our platform.',
     *   html: '<h1>Welcome!</h1><p>Welcome to our platform.</p>',
     * });
     * console.log('Sent:', result.id, 'Status:', result.status);
     * ```
     */
    send(params: EmailSendParams): Promise<EmailOutbound>;
    /**
     * List inbound emails
     *
     * @param addressId - optional email address ID to filter by
     * @returns array of inbound emails
     *
     * @example
     * ```typescript
     * const inbound = await email.listInbound('addr_123');
     * for (const msg of inbound) {
     *   console.log(`From: ${msg.from}, Subject: ${msg.subject}`);
     * }
     * ```
     */
    listInbound(addressId?: string): Promise<EmailInbound[]>;
    /**
     * Get an inbound email by ID
     *
     * @param id - the inbound email ID
     * @returns the inbound email or null if not found
     *
     * @example
     * ```typescript
     * const msg = await email.getInbound('inb_123');
     * if (msg) {
     *   console.log('Subject:', msg.subject);
     * }
     * ```
     */
    getInbound(id: string): Promise<EmailInbound | null>;
    /**
     * Delete an inbound email by ID
     *
     * @param id - the inbound email ID
     *
     * @example
     * ```typescript
     * await email.deleteInbound('inb_abc123');
     * ```
     */
    deleteInbound(id: string): Promise<void>;
    /**
     * List outbound emails
     *
     * @param addressId - optional email address ID to filter by
     * @returns array of outbound emails
     *
     * @example
     * ```typescript
     * const outbound = await email.listOutbound('addr_123');
     * for (const msg of outbound) {
     *   console.log(`To: ${msg.to}, Status: ${msg.status}`);
     * }
     * ```
     */
    listOutbound(addressId?: string): Promise<EmailOutbound[]>;
    /**
     * Get an outbound email by ID
     *
     * @param id - the outbound email ID
     * @returns the outbound email or null if not found
     *
     * @example
     * ```typescript
     * const msg = await email.getOutbound('out_123');
     * if (msg) {
     *   console.log('Status:', msg.status);
     * }
     * ```
     */
    getOutbound(id: string): Promise<EmailOutbound | null>;
    /**
     * Delete an outbound email by ID
     *
     * @param id - the outbound email ID
     *
     * @example
     * ```typescript
     * await email.deleteOutbound('out_abc123');
     * ```
     */
    deleteOutbound(id: string): Promise<void>;
    /**
     * Get email activity time-series data
     *
     * @param params - optional parameters (days defaults to 7)
     * @returns activity data points and the number of days queried
     *
     * @example
     * ```typescript
     * const activity = await email.getActivity({ days: 30 });
     * for (const point of activity.activity) {
     *   console.log(`${point.date}: ${point.inbound} in, ${point.outbound} out`);
     * }
     * ```
     */
    getActivity(params?: EmailActivityParams): Promise<EmailActivityResult>;
}
/**
 * Client for the Agentuity Email service.
 *
 * Provides methods for managing email addresses, configuring inbound email
 * destinations, sending outbound emails, and querying email history.
 *
 * Email addresses are created under the `@agentuity.email` domain. Inbound emails
 * can be forwarded to URL destinations. Outbound emails are sent asynchronously
 * and support attachments up to 25 MB total.
 *
 * All methods are instrumented with OpenTelemetry spans for observability.
 *
 * @example
 * ```typescript
 * const email = new EmailStorageService(baseUrl, adapter);
 *
 * // Create an address
 * const addr = await email.createAddress('notifications');
 *
 * // Send an email
 * await email.send({
 *   from: addr.email,
 *   to: ['user@example.com'],
 *   subject: 'Hello',
 *   text: 'Hello from Agentuity!',
 * });
 * ```
 */
export declare class EmailStorageService implements EmailService {
    #private;
    /**
     * Create a new EmailStorageService instance.
     *
     * @param baseUrl - The base URL for the Agentuity Email API (e.g., `https://api.agentuity.com`)
     * @param adapter - The HTTP fetch adapter used for making API requests
     */
    constructor(baseUrl: string, adapter: FetchAdapter);
    /**
     * Create a new email address under the `@agentuity.email` domain.
     *
     * @param localPart - The local part of the email address (the part before the `@`).
     *   For example, passing `'support'` creates `support@agentuity.email`.
     * @returns The newly created email address record
     * @throws ServiceException on API errors (e.g., duplicate address, invalid local part)
     *
     * @example
     * ```typescript
     * const addr = await email.createAddress('support');
     * console.log('Created:', addr.email); // support@agentuity.email
     * ```
     */
    createAddress(localPart: string): Promise<EmailAddress>;
    /**
     * List all email addresses owned by the current organization.
     *
     * @returns An array of email address records. Returns an empty array if none exist.
     * @throws ServiceException on API errors
     *
     * @example
     * ```typescript
     * const addresses = await email.listAddresses();
     * for (const addr of addresses) {
     *   console.log(`${addr.email} — ${addr.inbound_count ?? 0} received`);
     * }
     * ```
     */
    listAddresses(): Promise<EmailAddress[]>;
    /**
     * Get an email address by its ID.
     *
     * @param id - The email address ID (prefixed with `eaddr_`)
     * @returns The email address record, or `null` if no address with the given ID exists
     * @throws ServiceException on API errors (other than 404)
     *
     * @example
     * ```typescript
     * const addr = await email.getAddress('eaddr_abc123');
     * if (addr) {
     *   console.log('Found:', addr.email);
     * }
     * ```
     */
    getAddress(id: string): Promise<EmailAddress | null>;
    /**
     * Get IMAP and POP3 connection settings for an email address.
     *
     * These settings can be used to configure an external mail client (e.g., Thunderbird, Outlook)
     * to access the mailbox associated with the given address.
     *
     * @param id - The email address ID (prefixed with `eaddr_`)
     * @returns The connection configuration with IMAP and POP3 settings, or `null` if the address is not found
     * @throws ServiceException on API errors (other than 404)
     *
     * @example
     * ```typescript
     * const config = await email.getConnectionConfig('eaddr_abc123');
     * if (config) {
     *   console.log('IMAP host:', config.imap.host);
     *   console.log('POP3 host:', config.pop3.host);
     * }
     * ```
     */
    getConnectionConfig(id: string): Promise<EmailConnectionConfig | null>;
    /**
     * Delete an email address and all associated destinations.
     *
     * @remarks This operation is idempotent — deleting a non-existent address does not throw.
     *
     * @param id - The email address ID (prefixed with `eaddr_`)
     * @throws ServiceException on API errors (other than 404)
     *
     * @example
     * ```typescript
     * await email.deleteAddress('eaddr_abc123');
     * ```
     */
    deleteAddress(id: string): Promise<void>;
    /**
     * Create a new destination for an email address.
     *
     * Destinations determine where inbound emails are forwarded when they arrive
     * at the parent address.
     *
     * @param addressId - The email address ID (prefixed with `eaddr_`)
     * @param type - The destination type (currently only `'url'` is supported)
     * @param config - Type-specific destination configuration. For `'url'`:
     *   `{ url: string, headers?: Record<string, string>, method?: 'POST' | 'PUT' | 'PATCH' }`
     * @returns The newly created destination record
     * @throws ServiceException on API errors (e.g., invalid URL, address not found)
     *
     * @example
     * ```typescript
     * const dest = await email.createDestination('eaddr_abc123', 'url', {
     *   url: 'https://example.com/webhook',
     *   headers: { 'X-Secret': 'my-token' },
     * });
     * console.log('Destination created:', dest.id);
     * ```
     */
    createDestination(addressId: string, type: string, config: Record<string, unknown>): Promise<EmailDestination>;
    /**
     * List all destinations configured for an email address.
     *
     * @param addressId - The email address ID (prefixed with `eaddr_`)
     * @returns An array of destination records. Returns an empty array if none exist.
     * @throws ServiceException on API errors
     *
     * @example
     * ```typescript
     * const destinations = await email.listDestinations('eaddr_abc123');
     * for (const dest of destinations) {
     *   console.log(`${dest.type}: ${dest.id}`);
     * }
     * ```
     */
    listDestinations(addressId: string): Promise<EmailDestination[]>;
    /**
     * Delete a destination from an email address.
     *
     * @remarks This operation is idempotent — deleting a non-existent destination does not throw.
     *
     * @param addressId - The email address ID (prefixed with `eaddr_`)
     * @param destinationId - The destination ID (prefixed with `edst_`)
     * @throws ServiceException on API errors (other than 404)
     *
     * @example
     * ```typescript
     * await email.deleteDestination('eaddr_abc123', 'edst_xyz789');
     * ```
     */
    deleteDestination(addressId: string, destinationId: string): Promise<void>;
    /**
     * Send an outbound email from an Agentuity email address.
     *
     * Emails are sent asynchronously — this method returns immediately with an outbound
     * record whose status is `'pending'`. Use {@link getOutbound} to poll for delivery status.
     *
     * @remarks
     * - The `from` address must be owned by the current organization.
     * - Maximum 50 recipients per send.
     * - Maximum 25 MB for the full RFC 822 body (including attachments).
     *
     * @param params - The email send parameters including from, to, subject, and body
     * @returns The outbound email record with initial status `'pending'`
     * @throws ServiceException on API errors (e.g., invalid sender, too many recipients)
     *
     * @example
     * ```typescript
     * const result = await email.send({
     *   from: 'notifications@agentuity.email',
     *   to: ['user@example.com'],
     *   subject: 'Welcome!',
     *   text: 'Welcome to our platform.',
     *   html: '<h1>Welcome!</h1>',
     *   attachments: [{
     *     filename: 'guide.pdf',
     *     content: base64EncodedPdf,
     *     contentType: 'application/pdf',
     *   }],
     * });
     * console.log('Email queued:', result.id);
     * ```
     */
    send(params: EmailSendParams): Promise<EmailOutbound>;
    /**
     * List inbound emails, optionally filtered by email address.
     *
     * @param addressId - Optional email address ID (prefixed with `eaddr_`) to filter results.
     *   When omitted, returns inbound emails across all addresses in the organization.
     * @returns An array of inbound email records. Returns an empty array if none exist.
     * @throws ServiceException on API errors
     *
     * @example
     * ```typescript
     * // List all inbound emails
     * const all = await email.listInbound();
     *
     * // List inbound for a specific address
     * const filtered = await email.listInbound('eaddr_abc123');
     * for (const msg of filtered) {
     *   console.log(`From: ${msg.from}, Subject: ${msg.subject}`);
     * }
     * ```
     */
    listInbound(addressId?: string): Promise<EmailInbound[]>;
    /**
     * Get an inbound email by its ID.
     *
     * @param id - The inbound email ID (prefixed with `einb_`)
     * @returns The inbound email record, or `null` if not found
     * @throws ServiceException on API errors (other than 404)
     *
     * @example
     * ```typescript
     * const msg = await email.getInbound('einb_abc123');
     * if (msg) {
     *   console.log('Subject:', msg.subject);
     *   console.log('Attachments:', msg.attachments?.length ?? 0);
     * }
     * ```
     */
    getInbound(id: string): Promise<EmailInbound | null>;
    /**
     * Delete an inbound email by its ID.
     *
     * @remarks This operation is idempotent — deleting a non-existent email does not throw.
     *
     * @param id - The inbound email ID (prefixed with `einb_`)
     * @throws ServiceException on API errors (other than 404)
     *
     * @example
     * ```typescript
     * await email.deleteInbound('einb_abc123');
     * ```
     */
    deleteInbound(id: string): Promise<void>;
    /**
     * List outbound emails, optionally filtered by email address.
     *
     * @param addressId - Optional email address ID (prefixed with `eaddr_`) to filter results.
     *   When omitted, returns outbound emails across all addresses in the organization.
     * @returns An array of outbound email records. Returns an empty array if none exist.
     * @throws ServiceException on API errors
     *
     * @example
     * ```typescript
     * // List all outbound emails
     * const all = await email.listOutbound();
     *
     * // List outbound for a specific address
     * const filtered = await email.listOutbound('eaddr_abc123');
     * for (const msg of filtered) {
     *   console.log(`To: ${msg.to}, Status: ${msg.status}`);
     * }
     * ```
     */
    listOutbound(addressId?: string): Promise<EmailOutbound[]>;
    /**
     * Get an outbound email by its ID.
     *
     * @param id - The outbound email ID (prefixed with `eout_`)
     * @returns The outbound email record, or `null` if not found
     * @throws ServiceException on API errors (other than 404)
     *
     * @example
     * ```typescript
     * const msg = await email.getOutbound('eout_abc123');
     * if (msg) {
     *   console.log('Status:', msg.status);
     *   if (msg.error) {
     *     console.error('Delivery failed:', msg.error);
     *   }
     * }
     * ```
     */
    getOutbound(id: string): Promise<EmailOutbound | null>;
    /**
     * Delete an outbound email by its ID.
     *
     * @remarks This operation is idempotent — deleting a non-existent email does not throw.
     *
     * @param id - The outbound email ID (prefixed with `eout_`)
     * @throws ServiceException on API errors (other than 404)
     *
     * @example
     * ```typescript
     * await email.deleteOutbound('eout_abc123');
     * ```
     */
    deleteOutbound(id: string): Promise<void>;
    /**
     * Get email activity time-series data showing inbound and outbound counts per day.
     *
     * @param params - Optional query parameters. `days` controls the lookback window
     *   (minimum 7, maximum 365, server default 7).
     * @returns An {@link EmailActivityResult} with daily data points ordered chronologically
     *   and the total number of days returned
     * @throws ServiceException on API errors
     *
     * @example
     * ```typescript
     * // Get last 30 days of activity
     * const result = await email.getActivity({ days: 30 });
     * console.log(`Activity over ${result.days} days:`);
     * for (const point of result.activity) {
     *   console.log(`  ${point.date}: ${point.inbound} in, ${point.outbound} out`);
     * }
     * ```
     */
    getActivity(params?: EmailActivityParams): Promise<EmailActivityResult>;
}
//# sourceMappingURL=email.d.ts.map