import { APIResponseSchema } from '@agentuity/server';
import { z } from 'zod';
import { StructuredError } from '@agentuity/core';
import { createPublicKey } from 'crypto';
const DevmodeRequestSchema = z.object({
    hostname: z.string().optional().describe('the hostname for the endpoint'),
    publicKey: z.string().optional().describe('the public key PEM for the endpoint'),
});
function extractPublicKeyPEM(privateKeyPEM) {
    try {
        const publicKey = createPublicKey(privateKeyPEM);
        return publicKey.export({ type: 'spki', format: 'pem' });
    }
    catch {
        return undefined;
    }
}
const DevmodeResponseSchema = z.object({
    id: z.string(),
    hostname: z.string(),
    privateKey: z.string().optional(),
});
const DevmodeResponseAPISchema = APIResponseSchema(DevmodeResponseSchema);
const DevmodeEndpointError = StructuredError('DevmodeEndpointError');
/**
 * Generate an Endpoint ID and Hostname
 *
 * @param apiClient the api client to use
 * @param projectId the project id
 * @param hostname the hostname is already configured
 * @param privateKey the private key PEM if already configured
 * @returns
 */
export async function generateEndpoint(apiClient, projectId, hostname, privateKey) {
    const publicKey = privateKey ? extractPublicKeyPEM(privateKey) : undefined;
    const resp = await apiClient.request('POST', `/cli/devmode/3/${projectId}`, DevmodeResponseAPISchema, { hostname, publicKey }, DevmodeRequestSchema);
    if (!resp.success) {
        throw new DevmodeEndpointError({ message: resp.message });
    }
    return resp.data;
}
//# sourceMappingURL=api.js.map