import { z } from 'zod';
import { APIResponseSchema } from "../api.js";
import { SandboxResponseError } from "./util.js";
export const SandboxInfoSchema = z.object({
    id: z.string().describe('the sandbox id'),
    name: z.string().nullable().describe('the sandbox name'),
    description: z.string().nullable().describe('the sandbox description'),
    status: z.string().describe('sandbox status'),
    region: z.string().nullable().describe('cloud region'),
    createdAt: z.string().nullable().describe('ISO 8601 creation timestamp'),
    orgId: z.string().describe('the organization id'),
    orgName: z.string().nullable().describe('the organization name'),
    projectId: z.string().nullable().describe('the project id'),
});
export const SandboxListDataSchema = z.object({
    sandboxes: z.array(SandboxInfoSchema).describe('list of sandboxes'),
    total: z.number().describe('total count of matching sandboxes'),
});
export const SandboxListResponseSchema = APIResponseSchema(SandboxListDataSchema);
/**
 * List sandboxes with optional filtering using the CLI API endpoint.
 *
 * This endpoint searches across all organizations the user is a member of,
 * unlike the Catalyst API which requires an orgId.
 *
 * @param client - The API client
 * @param options - Filtering and pagination options
 * @returns A promise that resolves to the list of sandboxes
 *
 * @example
 * // List all sandboxes across all orgs
 * const result = await cliSandboxList(client);
 * console.log(`Found ${result.total} sandboxes`);
 *
 * @example
 * // List running sandboxes
 * const result = await cliSandboxList(client, { status: 'running' });
 *
 * @example
 * // List sandboxes for a specific project
 * const result = await cliSandboxList(client, { projectId: 'proj_123' });
 */
export async function cliSandboxList(client, options = {}) {
    const { projectId, orgId, status, limit, offset, sort, direction } = options;
    const params = new URLSearchParams();
    if (options.name)
        params.set('name', options.name);
    if (options.mode)
        params.set('mode', options.mode);
    if (projectId)
        params.set('projectId', projectId);
    if (orgId)
        params.set('orgId', orgId);
    if (status)
        params.set('status', status);
    if (limit !== undefined)
        params.set('limit', limit.toString());
    if (offset !== undefined)
        params.set('offset', offset.toString());
    if (sort)
        params.set('sort', sort);
    if (direction)
        params.set('direction', direction);
    const queryString = params.toString();
    const resp = await client.request('GET', `/cli/sandbox${queryString ? `?${queryString}` : ''}`, SandboxListResponseSchema);
    if (resp.success) {
        return resp.data;
    }
    throw new SandboxResponseError({ message: resp.message });
}
//# sourceMappingURL=cli-list.js.map