import { z } from 'zod';
import { type APIClient } from '../api.ts';
import { type CreateDestinationRequest, type Destination, type QueueApiOptions, type UpdateDestinationRequest } from './types.ts';
export declare const DestinationResponseSchema: z.ZodDiscriminatedUnion<[z.ZodObject<{
    success: z.ZodLiteral<false>;
    message: z.ZodString;
    code: z.ZodOptional<z.ZodString>;
}, z.core.$strip>, z.ZodObject<{
    success: z.ZodLiteral<true>;
    data: z.ZodObject<{
        destination: z.ZodObject<{
            id: z.ZodString;
            queue_id: z.ZodString;
            destination_type: z.ZodEnum<{
                http: "http";
            }>;
            config: z.ZodObject<{
                url: z.ZodString;
                headers: z.ZodOptional<z.ZodRecord<z.ZodString, z.ZodString>>;
                method: z.ZodDefault<z.ZodString>;
                timeout_ms: z.ZodDefault<z.ZodNumber>;
                retry_policy: z.ZodOptional<z.ZodObject<{
                    max_attempts: z.ZodDefault<z.ZodNumber>;
                    initial_backoff_ms: z.ZodDefault<z.ZodNumber>;
                    max_backoff_ms: z.ZodDefault<z.ZodNumber>;
                    backoff_multiplier: z.ZodDefault<z.ZodNumber>;
                }, z.core.$strip>>;
                signing: z.ZodOptional<z.ZodObject<{
                    enabled: z.ZodDefault<z.ZodBoolean>;
                    secret_key: z.ZodOptional<z.ZodString>;
                }, z.core.$strip>>;
            }, z.core.$strip>;
            enabled: z.ZodBoolean;
            stats: z.ZodOptional<z.ZodObject<{
                total_deliveries: z.ZodNumber;
                successful_deliveries: z.ZodNumber;
                failed_deliveries: z.ZodNumber;
                last_delivery_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
                last_success_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
                last_failure_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
            }, z.core.$strip>>;
            created_at: z.ZodString;
            updated_at: z.ZodString;
        }, z.core.$strip>;
    }, z.core.$strip>;
}, z.core.$strip>], "success">;
export declare const DestinationsListResponseSchema: z.ZodDiscriminatedUnion<[z.ZodObject<{
    success: z.ZodLiteral<false>;
    message: z.ZodString;
    code: z.ZodOptional<z.ZodString>;
}, z.core.$strip>, z.ZodObject<{
    success: z.ZodLiteral<true>;
    data: z.ZodObject<{
        destinations: z.ZodArray<z.ZodObject<{
            id: z.ZodString;
            queue_id: z.ZodString;
            destination_type: z.ZodEnum<{
                http: "http";
            }>;
            config: z.ZodObject<{
                url: z.ZodString;
                headers: z.ZodOptional<z.ZodRecord<z.ZodString, z.ZodString>>;
                method: z.ZodDefault<z.ZodString>;
                timeout_ms: z.ZodDefault<z.ZodNumber>;
                retry_policy: z.ZodOptional<z.ZodObject<{
                    max_attempts: z.ZodDefault<z.ZodNumber>;
                    initial_backoff_ms: z.ZodDefault<z.ZodNumber>;
                    max_backoff_ms: z.ZodDefault<z.ZodNumber>;
                    backoff_multiplier: z.ZodDefault<z.ZodNumber>;
                }, z.core.$strip>>;
                signing: z.ZodOptional<z.ZodObject<{
                    enabled: z.ZodDefault<z.ZodBoolean>;
                    secret_key: z.ZodOptional<z.ZodString>;
                }, z.core.$strip>>;
            }, z.core.$strip>;
            enabled: z.ZodBoolean;
            stats: z.ZodOptional<z.ZodObject<{
                total_deliveries: z.ZodNumber;
                successful_deliveries: z.ZodNumber;
                failed_deliveries: z.ZodNumber;
                last_delivery_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
                last_success_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
                last_failure_at: z.ZodOptional<z.ZodNullable<z.ZodString>>;
            }, z.core.$strip>>;
            created_at: z.ZodString;
            updated_at: z.ZodString;
        }, z.core.$strip>>;
    }, z.core.$strip>;
}, z.core.$strip>], "success">;
export declare const DeleteDestinationResponseSchema: z.ZodDiscriminatedUnion<[z.ZodObject<{
    success: z.ZodLiteral<false>;
    message: z.ZodString;
    code: z.ZodOptional<z.ZodString>;
}, z.core.$strip>, z.ZodObject<{
    success: z.ZodLiteral<true>;
}, z.core.$strip>], "success">;
/**
 * Create a destination for a queue.
 *
 * Destinations are webhook endpoints that automatically receive messages when
 * they are published to a queue. When a message is published, it will be
 * delivered via HTTP POST to all active destinations configured for that queue.
 *
 * @param client - The API client instance
 * @param queueName - The name of the queue to add the destination to
 * @param params - Destination configuration including URL and optional settings
 * @returns The created destination with assigned ID
 * @throws {QueueValidationError} If validation fails (invalid queue name or config)
 * @throws {QueueNotFoundError} If the queue does not exist
 * @throws {QueueError} If the API request fails
 *
 * @example
 * ```typescript
 * const destination = await createDestination(client, 'order-events', {
 *   url: 'https://api.example.com/webhooks/orders',
 *   config: {
 *     retry_attempts: 3,
 *     timeout_seconds: 30,
 *   },
 * });
 * console.log(`Created destination ${destination.id}`);
 * ```
 */
export declare function createDestination(client: APIClient, queueName: string, params: CreateDestinationRequest, options?: QueueApiOptions): Promise<Destination>;
/**
 * List all destinations for a queue.
 *
 * Retrieves all webhook destinations configured for a queue. Each destination
 * represents an endpoint that receives messages when they are published.
 *
 * @param client - The API client instance
 * @param queueName - The name of the queue
 * @returns Array of destinations configured for the queue
 * @throws {QueueValidationError} If validation fails (invalid queue name)
 * @throws {QueueNotFoundError} If the queue does not exist
 * @throws {QueueError} If the API request fails
 *
 * @example
 * ```typescript
 * const destinations = await listDestinations(client, 'order-events');
 * for (const dest of destinations) {
 *   console.log(`Destination ${dest.id}: ${dest.url} (${dest.enabled ? 'enabled' : 'disabled'})`);
 * }
 * ```
 */
export declare function listDestinations(client: APIClient, queueName: string, options?: QueueApiOptions): Promise<Destination[]>;
/**
 * Update a destination's configuration.
 *
 * Modifies an existing destination's settings such as URL, enabled status,
 * or retry configuration. Only the fields provided in params will be updated.
 *
 * @param client - The API client instance
 * @param queueName - The name of the queue
 * @param destinationId - The destination ID to update (prefixed with dest_)
 * @param params - Fields to update (partial update supported)
 * @returns The updated destination
 * @throws {QueueValidationError} If validation fails (invalid queue name, destination ID, or config)
 * @throws {DestinationNotFoundError} If the destination does not exist
 * @throws {QueueNotFoundError} If the queue does not exist
 * @throws {QueueError} If the API request fails
 *
 * @example
 * ```typescript
 * // Disable a destination temporarily
 * const updated = await updateDestination(client, 'order-events', 'dest_abc123', {
 *   enabled: false,
 * });
 *
 * // Update URL and retry settings
 * const updated = await updateDestination(client, 'order-events', 'dest_abc123', {
 *   url: 'https://api.example.com/v2/webhooks/orders',
 *   config: {
 *     retry_attempts: 5,
 *   },
 * });
 * ```
 */
export declare function updateDestination(client: APIClient, queueName: string, destinationId: string, params: UpdateDestinationRequest, options?: QueueApiOptions): Promise<Destination>;
/**
 * Delete a destination from a queue.
 *
 * Permanently removes a webhook destination. Messages will no longer be
 * delivered to this endpoint. This action cannot be undone.
 *
 * @param client - The API client instance
 * @param queueName - The name of the queue
 * @param destinationId - The destination ID to delete (prefixed with dest_)
 * @returns void
 * @throws {QueueValidationError} If validation fails (invalid queue name or destination ID)
 * @throws {DestinationNotFoundError} If the destination does not exist
 * @throws {QueueNotFoundError} If the queue does not exist
 * @throws {QueueError} If the API request fails
 *
 * @example
 * ```typescript
 * await deleteDestination(client, 'order-events', 'dest_abc123');
 * console.log('Destination deleted');
 * ```
 */
export declare function deleteDestination(client: APIClient, queueName: string, destinationId: string, options?: QueueApiOptions): Promise<void>;
//# sourceMappingURL=destinations.d.ts.map