import { createIssue, failure, success, createParseMethods, SCHEMA_KIND } from "../base.js";
import { optional } from "../utils/optional.js";
import { nullable } from "../utils/nullable.js";
const parseMethods = createParseMethods();
/**
 * Schema for validating number values.
 * Rejects NaN values.
 *
 * @example
 * ```typescript
 * const schema = s.number();
 * const age = schema.parse(30); // 30
 * schema.parse('30'); // throws ValidationError
 * schema.parse(NaN); // throws ValidationError
 * ```
 */
export class NumberSchema {
    [SCHEMA_KIND] = 'NumberSchema';
    description;
    _finite = false;
    _min;
    _max;
    '~standard' = {
        version: 1,
        vendor: 'agentuity',
        validate: (value) => {
            if (typeof value !== 'number' || Number.isNaN(value)) {
                return failure([createIssue(`Expected number, got ${typeof value}`)]);
            }
            if (this._finite && !Number.isFinite(value)) {
                return failure([createIssue('Expected finite number (not Infinity or -Infinity)')]);
            }
            if (this._min !== undefined && value < this._min) {
                return failure([createIssue(`Expected number >= ${this._min}, got ${value}`)]);
            }
            if (this._max !== undefined && value > this._max) {
                return failure([createIssue(`Expected number <= ${this._max}, got ${value}`)]);
            }
            return success(value);
        },
        types: undefined,
    };
    describe(description) {
        this.description = description;
        return this;
    }
    /**
     * Require the number to be finite (not Infinity, -Infinity, or NaN).
     *
     * @example
     * ```typescript
     * const schema = s.number().finite();
     * schema.parse(123); // 123
     * schema.parse(Infinity); // throws ValidationError
     * schema.parse(-Infinity); // throws ValidationError
     * ```
     */
    finite() {
        const clone = this._clone();
        clone._finite = true;
        return clone;
    }
    /**
     * Set minimum value (inclusive).
     *
     * @example
     * ```typescript
     * const schema = s.number().min(0);
     * schema.parse(5); // 5
     * schema.parse(-1); // throws ValidationError
     * ```
     */
    min(value) {
        const clone = this._clone();
        clone._min = value;
        return clone;
    }
    /**
     * Set maximum value (inclusive).
     *
     * @example
     * ```typescript
     * const schema = s.number().max(100);
     * schema.parse(50); // 50
     * schema.parse(101); // throws ValidationError
     * ```
     */
    max(value) {
        const clone = this._clone();
        clone._max = value;
        return clone;
    }
    optional() {
        return optional(this);
    }
    nullable() {
        return nullable(this);
    }
    _clone() {
        const clone = new NumberSchema();
        clone.description = this.description;
        clone._finite = this._finite;
        clone._min = this._min;
        clone._max = this._max;
        return clone;
    }
    parse = parseMethods.parse;
    safeParse = parseMethods.safeParse;
}
/**
 * Create a number schema.
 *
 * @example
 * ```typescript
 * const ageSchema = s.number().describe('User age');
 * const age = ageSchema.parse(30);
 *
 * const finiteSchema = s.number().finite();
 * finiteSchema.parse(123); // OK
 * finiteSchema.parse(Infinity); // throws
 *
 * const rangeSchema = s.number().min(0).max(100);
 * rangeSchema.parse(50); // OK
 * rangeSchema.parse(101); // throws
 * ```
 */
export function number() {
    return new NumberSchema();
}
//# sourceMappingURL=number.js.map