/**
 * Strips leading whitespace and SQL comments (block and line) from a query string.
 * Returns the remaining query text starting at the first non-comment token.
 *
 * Note: This regex does NOT support nested block comments. Use
 * {@link stripLeadingComments} for full nested comment support.
 */
export declare const LEADING_COMMENTS_RE: RegExp;
/**
 * Determines whether a SQL query is a mutation that requires transaction
 * wrapping for safe retry.
 *
 * Detected mutation types: INSERT, UPDATE, DELETE, COPY, TRUNCATE, MERGE,
 * CALL, DO. EXPLAIN queries are never wrapped (read-only analysis, even
 * when the explained statement is a mutation like `EXPLAIN INSERT INTO ...`).
 *
 * Mutation statements wrapped in a transaction can be safely retried because
 * PostgreSQL guarantees that uncommitted transactions are rolled back when
 * the connection drops. This prevents:
 * - Duplicate rows from retried INSERTs
 * - Double-applied changes from retried UPDATEs (e.g., counter increments)
 * - Repeated side effects from retried DELETEs (e.g., cascade triggers)
 *
 * Handles three patterns:
 * 1. Direct mutations: `INSERT INTO ...`, `UPDATE ... SET`, `DELETE FROM ...`,
 *    `COPY ...`, `TRUNCATE ...`, `MERGE ...`, `CALL ...`, `DO ...`
 *    (with optional leading comments/whitespace)
 * 2. CTE mutations: `WITH cte AS (...) INSERT|UPDATE|DELETE|... ...` — scans
 *    past the WITH clause by tracking parenthesis depth to skip CTE
 *    subexpressions, then checks if the first top-level DML keyword is a
 *    mutation. The scanner treats single-quoted strings, double-quoted
 *    identifiers, dollar-quoted strings, line comments (--), and block
 *    comments (including nested) as atomic regions so parentheses inside
 *    them do not corrupt depth tracking.
 * 3. Multi-statement queries: `SELECT 1; INSERT INTO items VALUES (1)` —
 *    scans past semicolons at depth 0 to find mutation keywords in
 *    subsequent statements.
 *
 * @see https://github.com/agentuity/sdk/issues/911
 */
export declare function isMutationStatement(query: string): boolean;
//# sourceMappingURL=mutation.d.ts.map