/**
 * @agentuity/postgres - Resilient PostgreSQL client with automatic reconnection
 *
 * This package provides a PostgreSQL client that wraps Bun's native SQL driver
 * and adds automatic reconnection with exponential backoff.
 *
 * @example
 * ```typescript
 * import { postgres } from '@agentuity/postgres';
 *
 * // Create a client (uses DATABASE_URL by default)
 * const sql = postgres();
 *
 * // Execute queries using tagged template literals
 * const users = await sql`SELECT * FROM users WHERE active = ${true}`;
 *
 * // Transactions
 * const tx = await sql.begin();
 * try {
 *   await tx`INSERT INTO users (name) VALUES (${name})`;
 *   await tx.commit();
 * } catch (error) {
 *   await tx.rollback();
 *   throw error;
 * }
 *
 * // Close when done
 * await sql.close();
 * ```
 *
 * @packageDocumentation
 */
export { postgres, default } from './postgres.ts';
export { PostgresClient, createCallableClient, createThenable, type CallablePostgresClient, } from './client.ts';
export { PostgresPool, Pool, createPool } from './pool.ts';
export { Transaction, Savepoint, ReservedConnection } from './transaction.ts';
export { patchBunSQL, isPatched, SQL } from './patch.ts';
export { injectSslMode } from './tls.ts';
export { isMutationStatement } from './mutation.ts';
export type { PostgresConfig, ReconnectConfig, ConnectionStats, TLSConfig, TransactionOptions, ReserveOptions, PoolConfig, PoolStats, PoolSSLConfig, } from './types.ts';
export { PostgresError, ConnectionClosedError, ReconnectFailedError, QueryTimeoutError, TransactionError, UnsupportedOperationError, isRetryableError, } from './errors.ts';
export { computeBackoff, sleep, mergeReconnectConfig, DEFAULT_RECONNECT_CONFIG, } from './reconnect.ts';
export { shutdownAll, getClientCount, getClients, hasActiveClients, type Registrable, } from './registry.ts';
//# sourceMappingURL=index.d.ts.map