import { buildUrl, toServiceException } from "./_util.js";
import { StructuredError } from "../error.js";
function createTimeoutSignal(ms = 30_000) {
    if (typeof AbortSignal.timeout === 'function') {
        return AbortSignal.timeout(ms);
    }
    const controller = new AbortController();
    const timer = setTimeout(() => controller.abort(), ms);
    controller.signal.addEventListener('abort', () => clearTimeout(timer), { once: true });
    return controller.signal;
}
const WebhookResponseError = StructuredError('WebhookResponseError')();
export class WebhookService {
    #adapter;
    #baseUrl;
    constructor(baseUrl, adapter) {
        this.#adapter = adapter;
        this.#baseUrl = baseUrl;
    }
    #unwrap(raw) {
        if (raw !== null && typeof raw === 'object' && 'data' in raw) {
            return raw.data;
        }
        return raw;
    }
    async create(params) {
        const url = buildUrl(this.#baseUrl, '/webhook/2026-02-24/create');
        const signal = createTimeoutSignal();
        const res = await this.#adapter.invoke(url, {
            method: 'POST',
            signal,
            body: JSON.stringify(params),
            contentType: 'application/json',
            telemetry: {
                name: 'agentuity.webhook.create',
                attributes: {
                    name: params.name,
                },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return { webhook: res.data.data };
            }
            throw new WebhookResponseError({ status: res.response.status, message: res.data.message });
        }
        throw await toServiceException('POST', url, res.response);
    }
    async list(params) {
        const qs = new URLSearchParams();
        if (params?.limit !== undefined) {
            qs.set('limit', String(params.limit));
        }
        if (params?.offset !== undefined) {
            qs.set('offset', String(params.offset));
        }
        const path = qs.toString()
            ? `/webhook/2026-02-24/list?${qs.toString()}`
            : '/webhook/2026-02-24/list';
        const url = buildUrl(this.#baseUrl, path);
        const signal = createTimeoutSignal();
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.webhook.list',
                attributes: {
                    limit: String(params?.limit ?? ''),
                    offset: String(params?.offset ?? ''),
                },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                const unwrapped = this.#unwrap(res.data.data);
                if (Array.isArray(unwrapped)) {
                    return { webhooks: unwrapped, total: unwrapped.length };
                }
                const arr = Array.isArray(unwrapped.data) ? unwrapped.data : [];
                return { webhooks: arr, total: unwrapped.total ?? arr.length };
            }
            throw new WebhookResponseError({ status: res.response.status, message: res.data.message });
        }
        throw await toServiceException('GET', url, res.response);
    }
    async get(webhookId) {
        const url = buildUrl(this.#baseUrl, `/webhook/2026-02-24/get/${encodeURIComponent(webhookId)}`);
        const signal = createTimeoutSignal();
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.webhook.get',
                attributes: {
                    webhookId,
                },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                const { destinations } = await this.listDestinations(webhookId);
                return { webhook: res.data.data, destinations };
            }
            throw new WebhookResponseError({ status: res.response.status, message: res.data.message });
        }
        throw await toServiceException('GET', url, res.response);
    }
    async update(webhookId, params) {
        const url = buildUrl(this.#baseUrl, `/webhook/2026-02-24/update/${encodeURIComponent(webhookId)}`);
        const signal = createTimeoutSignal();
        const res = await this.#adapter.invoke(url, {
            method: 'PUT',
            signal,
            body: JSON.stringify(params),
            contentType: 'application/json',
            telemetry: {
                name: 'agentuity.webhook.update',
                attributes: {
                    webhookId,
                },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return { webhook: res.data.data };
            }
            throw new WebhookResponseError({ status: res.response.status, message: res.data.message });
        }
        throw await toServiceException('PUT', url, res.response);
    }
    async delete(webhookId) {
        const url = buildUrl(this.#baseUrl, `/webhook/2026-02-24/delete/${encodeURIComponent(webhookId)}`);
        const signal = createTimeoutSignal();
        const res = await this.#adapter.invoke(url, {
            method: 'DELETE',
            signal,
            telemetry: {
                name: 'agentuity.webhook.delete',
                attributes: {
                    webhookId,
                },
            },
        });
        if (res.ok) {
            if (res.data?.success !== false) {
                return;
            }
            throw new WebhookResponseError({
                status: res.response.status,
                message: res.data?.message ?? 'Delete failed',
            });
        }
        throw await toServiceException('DELETE', url, res.response);
    }
    async createDestination(webhookId, params) {
        const url = buildUrl(this.#baseUrl, `/webhook/2026-02-24/destination-create/${encodeURIComponent(webhookId)}`);
        const signal = createTimeoutSignal();
        const res = await this.#adapter.invoke(url, {
            method: 'POST',
            signal,
            body: JSON.stringify(params),
            contentType: 'application/json',
            telemetry: {
                name: 'agentuity.webhook.createDestination',
                attributes: {
                    webhookId,
                    type: params.type,
                },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return { destination: res.data.data };
            }
            throw new WebhookResponseError({ status: res.response.status, message: res.data.message });
        }
        throw await toServiceException('POST', url, res.response);
    }
    async listDestinations(webhookId) {
        const url = buildUrl(this.#baseUrl, `/webhook/2026-02-24/destination-list/${encodeURIComponent(webhookId)}`);
        const signal = createTimeoutSignal();
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.webhook.listDestinations',
                attributes: {
                    webhookId,
                },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return { destinations: Array.isArray(res.data.data) ? res.data.data : [] };
            }
            throw new WebhookResponseError({ status: res.response.status, message: res.data.message });
        }
        throw await toServiceException('GET', url, res.response);
    }
    async deleteDestination(webhookId, destinationId) {
        const url = buildUrl(this.#baseUrl, `/webhook/2026-02-24/destination-delete/${encodeURIComponent(webhookId)}/${encodeURIComponent(destinationId)}`);
        const signal = createTimeoutSignal();
        const res = await this.#adapter.invoke(url, {
            method: 'DELETE',
            signal,
            telemetry: {
                name: 'agentuity.webhook.deleteDestination',
                attributes: {
                    webhookId,
                    destinationId,
                },
            },
        });
        if (res.ok) {
            if (res.data?.success !== false) {
                return;
            }
            throw new WebhookResponseError({
                status: res.response.status,
                message: res.data?.message ?? 'Delete destination failed',
            });
        }
        throw await toServiceException('DELETE', url, res.response);
    }
    async listReceipts(webhookId, params) {
        const qs = new URLSearchParams();
        if (params?.limit !== undefined) {
            qs.set('limit', String(params.limit));
        }
        if (params?.offset !== undefined) {
            qs.set('offset', String(params.offset));
        }
        const basePath = `/webhook/2026-02-24/receipt-list/${encodeURIComponent(webhookId)}`;
        const path = qs.toString() ? `${basePath}?${qs.toString()}` : basePath;
        const url = buildUrl(this.#baseUrl, path);
        const signal = createTimeoutSignal();
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.webhook.listReceipts',
                attributes: {
                    webhookId,
                    limit: String(params?.limit ?? ''),
                    offset: String(params?.offset ?? ''),
                },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return { receipts: Array.isArray(res.data.data) ? res.data.data : [] };
            }
            throw new WebhookResponseError({ status: res.response.status, message: res.data.message });
        }
        throw await toServiceException('GET', url, res.response);
    }
    async getReceipt(webhookId, receiptId) {
        const url = buildUrl(this.#baseUrl, `/webhook/2026-02-24/receipt-get/${encodeURIComponent(webhookId)}/${encodeURIComponent(receiptId)}`);
        const signal = createTimeoutSignal();
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.webhook.getReceipt',
                attributes: {
                    webhookId,
                    receiptId,
                },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return res.data.data;
            }
            throw new WebhookResponseError({ status: res.response.status, message: res.data.message });
        }
        throw await toServiceException('GET', url, res.response);
    }
    async listDeliveries(webhookId, params) {
        const qs = new URLSearchParams();
        if (params?.limit !== undefined) {
            qs.set('limit', String(params.limit));
        }
        if (params?.offset !== undefined) {
            qs.set('offset', String(params.offset));
        }
        const basePath = `/webhook/2026-02-24/delivery-list/${encodeURIComponent(webhookId)}`;
        const path = qs.toString() ? `${basePath}?${qs.toString()}` : basePath;
        const url = buildUrl(this.#baseUrl, path);
        const signal = createTimeoutSignal();
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.webhook.listDeliveries',
                attributes: {
                    webhookId,
                    limit: String(params?.limit ?? ''),
                    offset: String(params?.offset ?? ''),
                },
            },
        });
        if (res.ok) {
            if (res.data.success) {
                return { deliveries: Array.isArray(res.data.data) ? res.data.data : [] };
            }
            throw new WebhookResponseError({ status: res.response.status, message: res.data.message });
        }
        throw await toServiceException('GET', url, res.response);
    }
    async retryDelivery(webhookId, deliveryId) {
        const url = buildUrl(this.#baseUrl, `/webhook/2026-02-24/delivery-retry/${encodeURIComponent(webhookId)}/${encodeURIComponent(deliveryId)}`);
        const signal = createTimeoutSignal();
        const res = await this.#adapter.invoke(url, {
            method: 'POST',
            signal,
            telemetry: {
                name: 'agentuity.webhook.retryDelivery',
                attributes: {
                    webhookId,
                    deliveryId,
                },
            },
        });
        if (res.ok) {
            if (res.data?.success !== false) {
                return;
            }
            throw new WebhookResponseError({
                status: res.response.status,
                message: res.data?.message ?? 'Retry delivery failed',
            });
        }
        throw await toServiceException('POST', url, res.response);
    }
}
//# sourceMappingURL=webhook.js.map