import { buildUrl, toServiceException } from "./_util.js";
export class ScheduleService {
    #adapter;
    #baseUrl;
    constructor(baseUrl, adapter) {
        this.#adapter = adapter;
        this.#baseUrl = baseUrl;
    }
    async create(params) {
        const url = buildUrl(this.#baseUrl, '/schedule/create/2026-02-24');
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'POST',
            signal,
            body: JSON.stringify(params),
            contentType: 'application/json',
            telemetry: {
                name: 'agentuity.schedule.create',
                attributes: {
                    destinationCount: String(params.destinations?.length ?? 0),
                    name: params.name,
                },
            },
        });
        if (res.ok) {
            return res.data;
        }
        throw await toServiceException('POST', url, res.response);
    }
    async list(params) {
        const qs = new URLSearchParams();
        if (params?.limit !== undefined) {
            qs.set('limit', String(params.limit));
        }
        if (params?.offset !== undefined) {
            qs.set('offset', String(params.offset));
        }
        const path = qs.toString()
            ? `/schedule/list/2026-02-24?${qs.toString()}`
            : '/schedule/list/2026-02-24';
        const url = buildUrl(this.#baseUrl, path);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.schedule.list',
                attributes: {
                    limit: String(params?.limit ?? ''),
                    offset: String(params?.offset ?? ''),
                },
            },
        });
        if (res.ok) {
            return res.data;
        }
        throw await toServiceException('GET', url, res.response);
    }
    async get(scheduleId) {
        const url = buildUrl(this.#baseUrl, `/schedule/get/2026-02-24/${encodeURIComponent(scheduleId)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.schedule.get',
                attributes: {
                    scheduleId,
                },
            },
        });
        if (res.ok) {
            return res.data;
        }
        if (res.response.status === 404) {
            throw await toServiceException('GET', url, res.response);
        }
        throw await toServiceException('GET', url, res.response);
    }
    async update(scheduleId, params) {
        const url = buildUrl(this.#baseUrl, `/schedule/update/2026-02-24/${encodeURIComponent(scheduleId)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'PUT',
            signal,
            body: JSON.stringify(params),
            contentType: 'application/json',
            telemetry: {
                name: 'agentuity.schedule.update',
                attributes: {
                    scheduleId,
                },
            },
        });
        if (res.ok) {
            return res.data;
        }
        throw await toServiceException('PUT', url, res.response);
    }
    async delete(scheduleId) {
        const url = buildUrl(this.#baseUrl, `/schedule/delete/2026-02-24/${encodeURIComponent(scheduleId)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'DELETE',
            signal,
            telemetry: {
                name: 'agentuity.schedule.delete',
                attributes: {
                    scheduleId,
                },
            },
        });
        if (res.ok) {
            return;
        }
        throw await toServiceException('DELETE', url, res.response);
    }
    async createDestination(scheduleId, params) {
        const url = buildUrl(this.#baseUrl, `/schedule/destinations/create/2026-02-24/${encodeURIComponent(scheduleId)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'POST',
            signal,
            body: JSON.stringify(params),
            contentType: 'application/json',
            telemetry: {
                name: 'agentuity.schedule.createDestination',
                attributes: {
                    scheduleId,
                    type: params.type,
                },
            },
        });
        if (res.ok) {
            return res.data;
        }
        throw await toServiceException('POST', url, res.response);
    }
    async deleteDestination(destinationId) {
        const url = buildUrl(this.#baseUrl, `/schedule/destinations/delete/2026-02-24/${encodeURIComponent(destinationId)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'DELETE',
            signal,
            telemetry: {
                name: 'agentuity.schedule.deleteDestination',
                attributes: {
                    destinationId,
                },
            },
        });
        if (res.ok) {
            return;
        }
        throw await toServiceException('DELETE', url, res.response);
    }
    async listDeliveries(scheduleId, params) {
        const qs = new URLSearchParams();
        if (params?.limit !== undefined) {
            qs.set('limit', String(params.limit));
        }
        if (params?.offset !== undefined) {
            qs.set('offset', String(params.offset));
        }
        const basePath = `/schedule/deliveries/2026-02-24/${encodeURIComponent(scheduleId)}`;
        const path = qs.toString() ? `${basePath}?${qs.toString()}` : basePath;
        const url = buildUrl(this.#baseUrl, path);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.schedule.listDeliveries',
                attributes: {
                    scheduleId,
                    limit: String(params?.limit ?? ''),
                    offset: String(params?.offset ?? ''),
                },
            },
        });
        if (res.ok) {
            return res.data;
        }
        if (res.response.status === 404) {
            throw await toServiceException('GET', url, res.response);
        }
        throw await toServiceException('GET', url, res.response);
    }
    async getDestination(scheduleId, destinationId) {
        const result = await this.get(scheduleId);
        const destination = result.destinations.find((d) => d.id === destinationId);
        if (!destination) {
            throw new Error(`Destination not found: ${destinationId}`);
        }
        return destination;
    }
    async getDelivery(scheduleId, deliveryId, params) {
        const result = await this.listDeliveries(scheduleId, params);
        const delivery = result.deliveries.find((d) => d.id === deliveryId);
        if (!delivery) {
            throw new Error(`Delivery not found: ${deliveryId}`);
        }
        return delivery;
    }
}
//# sourceMappingURL=schedule.js.map