/**
 * Vite plugin to fix incorrect public asset paths
 *
 * Developers should use /public/ paths for static assets from src/web/public/.
 * In production, these paths are transformed to CDN URLs.
 *
 * This plugin:
 * 1. During build: Rewrites /public/* paths to CDN URLs
 * 2. During dev: Warns only about incorrect source paths (src/web/public/)
 *
 * Supported patterns (work in dev, rewritten to CDN in production):
 * - '/public/foo.svg'          → CDN URL (recommended)
 * - './public/foo.svg'         → CDN URL
 * - 'url(/public/foo.svg)'    → CDN URL (CSS unquoted)
 * - 'url(./public/foo.svg)'   → CDN URL (CSS unquoted)
 *
 * Incorrect patterns (warned in dev, rewritten in production):
 * - '/src/web/public/foo.svg'  → CDN URL
 * - './src/web/public/foo.svg' → CDN URL
 * - 'src/web/public/foo.svg'   → CDN URL
 */
import type { Plugin } from 'vite';
export interface PublicAssetPathPluginOptions {
    /** Whether to show warnings in dev mode (default: true) */
    warnInDev?: boolean;
    /** CDN base URL for production builds (e.g., 'https://cdn.agentuity.com/{deploymentId}/client/') */
    cdnBaseUrl?: string;
}
/**
 * Vite plugin that fixes public asset paths and rewrites to CDN URLs
 *
 * Rewrites all public asset paths to CDN URLs in production.
 *
 * @example
 * // In vite config:
 * plugins: [publicAssetPathPlugin({ cdnBaseUrl: 'https://cdn.example.com/deploy/client/' })]
 *
 * // In code, use /public/ paths:
 * <img src="/public/logo.svg" />
 *
 * // Transforms in production:
 * // '/public/logo.svg' → 'https://cdn.example.com/deploy/client/logo.svg'
 */
export declare function publicAssetPathPlugin(options?: PublicAssetPathPluginOptions): Plugin;
//# sourceMappingURL=public-asset-path-plugin.d.ts.map