/**
 * Auth React integration for @agentuity/auth.
 *
 * All React-specific code for auth.
 * Import from '@agentuity/auth/react' for React components and hooks.
 *
 * @module agentuity/react
 */
import React from 'react';
import { createAuthClient as createBetterAuthClient } from 'better-auth/react';
import type { BetterAuthClientPlugin } from 'better-auth/client';
import type { AuthSession, AuthUser } from './types';
/**
 * Options for creating the auth client.
 *
 * @typeParam TPlugins - Array of BetterAuth client plugins for type inference
 */
export interface AuthClientOptions<TPlugins extends BetterAuthClientPlugin[] = BetterAuthClientPlugin[]> {
    /**
     * Base URL for auth API requests.
     * Defaults to `window.location.origin` in browser environments.
     */
    baseURL?: string;
    /**
     * Base path for auth endpoints.
     * Defaults to '/api/auth' (Agentuity convention).
     */
    basePath?: string;
    /**
     * Skip default plugins (organizationClient, apiKeyClient).
     * Use this if you want full control over plugins.
     */
    skipDefaultPlugins?: boolean;
    /**
     * Additional plugins to include.
     * These are added after the default plugins (unless skipDefaultPlugins is true).
     *
     * Plugin types are inferred for full type safety.
     */
    plugins?: TPlugins;
}
/**
 * Get the default client plugins for auth.
 *
 * These mirror the server-side plugins:
 * - organizationClient: Multi-tenancy support
 * - apiKeyClient: Programmatic API key management
 *
 * Note: jwt() and bearer() are server-only plugins.
 */
export declare function getDefaultClientPlugins(): ({
    id: "organization";
    $InferServerPlugin: import("better-auth/plugins").OrganizationPlugin<{
        ac: import("better-auth/plugins").AccessControl<{
            readonly organization: readonly ["update", "delete"];
            readonly member: readonly ["create", "update", "delete"];
            readonly invitation: readonly ["create", "cancel"];
            readonly team: readonly ["create", "update", "delete"];
            readonly ac: readonly ["create", "read", "update", "delete"];
        }>;
        roles: {
            admin: import("better-auth/plugins").Role;
            member: import("better-auth/plugins").Role;
            owner: import("better-auth/plugins").Role;
        };
        teams: {
            enabled: false;
        };
        schema: {
            organization?: {
                additionalFields?: {
                    [key: string]: import("better-auth").DBFieldAttribute;
                };
            };
            member?: {
                additionalFields?: {
                    [key: string]: import("better-auth").DBFieldAttribute;
                };
            };
            invitation?: {
                additionalFields?: {
                    [key: string]: import("better-auth").DBFieldAttribute;
                };
            };
            team?: {
                additionalFields?: {
                    [key: string]: import("better-auth").DBFieldAttribute;
                };
            };
            organizationRole?: {
                additionalFields?: {
                    [key: string]: import("better-auth").DBFieldAttribute;
                };
            };
        } | undefined;
        dynamicAccessControl: {
            enabled: false;
        };
    }>;
    getActions: ($fetch: import("better-auth/react").BetterFetch, _$store: import("better-auth").ClientStore, co: import("better-auth").BetterAuthClientOptions | undefined) => {
        $Infer: {
            ActiveOrganization: {
                members: {
                    id: string;
                    organizationId: string;
                    role: "member" | "admin" | "owner";
                    createdAt: Date;
                    userId: string;
                    user: {
                        id: string;
                        email: string;
                        name: string;
                        image?: string | undefined;
                    };
                }[];
                invitations: {
                    id: string;
                    organizationId: string;
                    email: string;
                    role: "member" | "admin" | "owner";
                    status: import("better-auth/plugins").InvitationStatus;
                    inviterId: string;
                    expiresAt: Date;
                    createdAt: Date;
                }[];
            } & {
                id: string;
                name: string;
                slug: string;
                createdAt: Date;
                logo?: string | null | undefined | undefined;
                metadata?: any;
            };
            Organization: {
                id: string;
                name: string;
                slug: string;
                createdAt: Date;
                logo?: string | null | undefined;
                metadata?: any;
            };
            Invitation: {
                id: string;
                organizationId: string;
                email: string;
                role: "member" | "admin" | "owner";
                status: import("better-auth/plugins").InvitationStatus;
                inviterId: string;
                expiresAt: Date;
                createdAt: Date;
            };
            Member: {
                id: string;
                organizationId: string;
                role: "member" | "admin" | "owner";
                createdAt: Date;
                userId: string;
                user: {
                    id: string;
                    email: string;
                    name: string;
                    image?: string | undefined;
                };
            };
            Team: {
                id: string;
                name: string;
                organizationId: string;
                createdAt: Date;
                updatedAt?: Date | undefined;
            };
        };
        organization: {
            checkRolePermission: <R extends "member" | "admin" | "owner">(data: {
                permissions: {
                    readonly organization?: ("update" | "delete")[] | undefined;
                    readonly member?: ("create" | "update" | "delete")[] | undefined;
                    readonly invitation?: ("create" | "cancel")[] | undefined;
                    readonly team?: ("create" | "update" | "delete")[] | undefined;
                    readonly ac?: ("create" | "update" | "delete" | "read")[] | undefined;
                };
            } & {
                role: R;
            }) => boolean;
        };
    };
    getAtoms: ($fetch: import("better-auth/react").BetterFetch) => {
        $listOrg: import("better-auth/react").PreinitializedWritableAtom<boolean> & object;
        $activeOrgSignal: import("better-auth/react").PreinitializedWritableAtom<boolean> & object;
        $activeMemberSignal: import("better-auth/react").PreinitializedWritableAtom<boolean> & object;
        $activeMemberRoleSignal: import("better-auth/react").PreinitializedWritableAtom<boolean> & object;
        activeOrganization: import("better-auth/client").AuthQueryAtom<import("better-auth").Prettify<{
            id: string;
            name: string;
            slug: string;
            createdAt: Date;
            logo?: string | null | undefined | undefined;
            metadata?: any;
        } & {
            members: {
                id: string;
                organizationId: string;
                role: "member" | "admin" | "owner";
                createdAt: Date;
                userId: string;
                user: {
                    id: string;
                    email: string;
                    name: string;
                    image?: string | undefined;
                };
            }[];
            invitations: {
                id: string;
                organizationId: string;
                email: string;
                role: "member" | "admin" | "owner";
                status: import("better-auth/plugins").InvitationStatus;
                inviterId: string;
                expiresAt: Date;
                createdAt: Date;
            }[];
        }>>;
        listOrganizations: import("better-auth/client").AuthQueryAtom<{
            id: string;
            name: string;
            slug: string;
            createdAt: Date;
            logo?: string | null | undefined | undefined;
            metadata?: any;
        }[]>;
        activeMember: import("better-auth/client").AuthQueryAtom<{
            id: string;
            organizationId: string;
            userId: string;
            role: string;
            createdAt: Date;
        }>;
        activeMemberRole: import("better-auth/client").AuthQueryAtom<{
            role: string;
        }>;
    };
    pathMethods: {
        "/organization/get-full-organization": "GET";
        "/organization/list-user-teams": "GET";
    };
    atomListeners: ({
        matcher(path: string): path is "/organization/create" | "/organization/update" | "/organization/delete";
        signal: "$listOrg";
    } | {
        matcher(path: string): boolean;
        signal: "$activeOrgSignal";
    } | {
        matcher(path: string): boolean;
        signal: "$sessionSignal";
    } | {
        matcher(path: string): boolean;
        signal: "$activeMemberSignal";
    } | {
        matcher(path: string): boolean;
        signal: "$activeMemberRoleSignal";
    })[];
    $ERROR_CODES: {
        YOU_ARE_NOT_ALLOWED_TO_CREATE_A_NEW_ORGANIZATION: import("better-auth").RawError<"YOU_ARE_NOT_ALLOWED_TO_CREATE_A_NEW_ORGANIZATION">;
        YOU_HAVE_REACHED_THE_MAXIMUM_NUMBER_OF_ORGANIZATIONS: import("better-auth").RawError<"YOU_HAVE_REACHED_THE_MAXIMUM_NUMBER_OF_ORGANIZATIONS">;
        ORGANIZATION_ALREADY_EXISTS: import("better-auth").RawError<"ORGANIZATION_ALREADY_EXISTS">;
        ORGANIZATION_SLUG_ALREADY_TAKEN: import("better-auth").RawError<"ORGANIZATION_SLUG_ALREADY_TAKEN">;
        ORGANIZATION_NOT_FOUND: import("better-auth").RawError<"ORGANIZATION_NOT_FOUND">;
        USER_IS_NOT_A_MEMBER_OF_THE_ORGANIZATION: import("better-auth").RawError<"USER_IS_NOT_A_MEMBER_OF_THE_ORGANIZATION">;
        YOU_ARE_NOT_ALLOWED_TO_UPDATE_THIS_ORGANIZATION: import("better-auth").RawError<"YOU_ARE_NOT_ALLOWED_TO_UPDATE_THIS_ORGANIZATION">;
        YOU_ARE_NOT_ALLOWED_TO_DELETE_THIS_ORGANIZATION: import("better-auth").RawError<"YOU_ARE_NOT_ALLOWED_TO_DELETE_THIS_ORGANIZATION">;
        NO_ACTIVE_ORGANIZATION: import("better-auth").RawError<"NO_ACTIVE_ORGANIZATION">;
        USER_IS_ALREADY_A_MEMBER_OF_THIS_ORGANIZATION: import("better-auth").RawError<"USER_IS_ALREADY_A_MEMBER_OF_THIS_ORGANIZATION">;
        MEMBER_NOT_FOUND: import("better-auth").RawError<"MEMBER_NOT_FOUND">;
        ROLE_NOT_FOUND: import("better-auth").RawError<"ROLE_NOT_FOUND">;
        YOU_ARE_NOT_ALLOWED_TO_CREATE_A_NEW_TEAM: import("better-auth").RawError<"YOU_ARE_NOT_ALLOWED_TO_CREATE_A_NEW_TEAM">;
        TEAM_ALREADY_EXISTS: import("better-auth").RawError<"TEAM_ALREADY_EXISTS">;
        TEAM_NOT_FOUND: import("better-auth").RawError<"TEAM_NOT_FOUND">;
        YOU_CANNOT_LEAVE_THE_ORGANIZATION_AS_THE_ONLY_OWNER: import("better-auth").RawError<"YOU_CANNOT_LEAVE_THE_ORGANIZATION_AS_THE_ONLY_OWNER">;
        YOU_CANNOT_LEAVE_THE_ORGANIZATION_WITHOUT_AN_OWNER: import("better-auth").RawError<"YOU_CANNOT_LEAVE_THE_ORGANIZATION_WITHOUT_AN_OWNER">;
        YOU_ARE_NOT_ALLOWED_TO_DELETE_THIS_MEMBER: import("better-auth").RawError<"YOU_ARE_NOT_ALLOWED_TO_DELETE_THIS_MEMBER">;
        YOU_ARE_NOT_ALLOWED_TO_INVITE_USERS_TO_THIS_ORGANIZATION: import("better-auth").RawError<"YOU_ARE_NOT_ALLOWED_TO_INVITE_USERS_TO_THIS_ORGANIZATION">;
        USER_IS_ALREADY_INVITED_TO_THIS_ORGANIZATION: import("better-auth").RawError<"USER_IS_ALREADY_INVITED_TO_THIS_ORGANIZATION">;
        INVITATION_NOT_FOUND: import("better-auth").RawError<"INVITATION_NOT_FOUND">;
        YOU_ARE_NOT_THE_RECIPIENT_OF_THE_INVITATION: import("better-auth").RawError<"YOU_ARE_NOT_THE_RECIPIENT_OF_THE_INVITATION">;
        EMAIL_VERIFICATION_REQUIRED_BEFORE_ACCEPTING_OR_REJECTING_INVITATION: import("better-auth").RawError<"EMAIL_VERIFICATION_REQUIRED_BEFORE_ACCEPTING_OR_REJECTING_INVITATION">;
        YOU_ARE_NOT_ALLOWED_TO_CANCEL_THIS_INVITATION: import("better-auth").RawError<"YOU_ARE_NOT_ALLOWED_TO_CANCEL_THIS_INVITATION">;
        INVITER_IS_NO_LONGER_A_MEMBER_OF_THE_ORGANIZATION: import("better-auth").RawError<"INVITER_IS_NO_LONGER_A_MEMBER_OF_THE_ORGANIZATION">;
        YOU_ARE_NOT_ALLOWED_TO_INVITE_USER_WITH_THIS_ROLE: import("better-auth").RawError<"YOU_ARE_NOT_ALLOWED_TO_INVITE_USER_WITH_THIS_ROLE">;
        FAILED_TO_RETRIEVE_INVITATION: import("better-auth").RawError<"FAILED_TO_RETRIEVE_INVITATION">;
        YOU_HAVE_REACHED_THE_MAXIMUM_NUMBER_OF_TEAMS: import("better-auth").RawError<"YOU_HAVE_REACHED_THE_MAXIMUM_NUMBER_OF_TEAMS">;
        UNABLE_TO_REMOVE_LAST_TEAM: import("better-auth").RawError<"UNABLE_TO_REMOVE_LAST_TEAM">;
        YOU_ARE_NOT_ALLOWED_TO_UPDATE_THIS_MEMBER: import("better-auth").RawError<"YOU_ARE_NOT_ALLOWED_TO_UPDATE_THIS_MEMBER">;
        ORGANIZATION_MEMBERSHIP_LIMIT_REACHED: import("better-auth").RawError<"ORGANIZATION_MEMBERSHIP_LIMIT_REACHED">;
        YOU_ARE_NOT_ALLOWED_TO_CREATE_TEAMS_IN_THIS_ORGANIZATION: import("better-auth").RawError<"YOU_ARE_NOT_ALLOWED_TO_CREATE_TEAMS_IN_THIS_ORGANIZATION">;
        YOU_ARE_NOT_ALLOWED_TO_DELETE_TEAMS_IN_THIS_ORGANIZATION: import("better-auth").RawError<"YOU_ARE_NOT_ALLOWED_TO_DELETE_TEAMS_IN_THIS_ORGANIZATION">;
        YOU_ARE_NOT_ALLOWED_TO_UPDATE_THIS_TEAM: import("better-auth").RawError<"YOU_ARE_NOT_ALLOWED_TO_UPDATE_THIS_TEAM">;
        YOU_ARE_NOT_ALLOWED_TO_DELETE_THIS_TEAM: import("better-auth").RawError<"YOU_ARE_NOT_ALLOWED_TO_DELETE_THIS_TEAM">;
        INVITATION_LIMIT_REACHED: import("better-auth").RawError<"INVITATION_LIMIT_REACHED">;
        TEAM_MEMBER_LIMIT_REACHED: import("better-auth").RawError<"TEAM_MEMBER_LIMIT_REACHED">;
        USER_IS_NOT_A_MEMBER_OF_THE_TEAM: import("better-auth").RawError<"USER_IS_NOT_A_MEMBER_OF_THE_TEAM">;
        YOU_CAN_NOT_ACCESS_THE_MEMBERS_OF_THIS_TEAM: import("better-auth").RawError<"YOU_CAN_NOT_ACCESS_THE_MEMBERS_OF_THIS_TEAM">;
        YOU_DO_NOT_HAVE_AN_ACTIVE_TEAM: import("better-auth").RawError<"YOU_DO_NOT_HAVE_AN_ACTIVE_TEAM">;
        YOU_ARE_NOT_ALLOWED_TO_CREATE_A_NEW_TEAM_MEMBER: import("better-auth").RawError<"YOU_ARE_NOT_ALLOWED_TO_CREATE_A_NEW_TEAM_MEMBER">;
        YOU_ARE_NOT_ALLOWED_TO_REMOVE_A_TEAM_MEMBER: import("better-auth").RawError<"YOU_ARE_NOT_ALLOWED_TO_REMOVE_A_TEAM_MEMBER">;
        YOU_ARE_NOT_ALLOWED_TO_ACCESS_THIS_ORGANIZATION: import("better-auth").RawError<"YOU_ARE_NOT_ALLOWED_TO_ACCESS_THIS_ORGANIZATION">;
        YOU_ARE_NOT_A_MEMBER_OF_THIS_ORGANIZATION: import("better-auth").RawError<"YOU_ARE_NOT_A_MEMBER_OF_THIS_ORGANIZATION">;
        MISSING_AC_INSTANCE: import("better-auth").RawError<"MISSING_AC_INSTANCE">;
        YOU_MUST_BE_IN_AN_ORGANIZATION_TO_CREATE_A_ROLE: import("better-auth").RawError<"YOU_MUST_BE_IN_AN_ORGANIZATION_TO_CREATE_A_ROLE">;
        YOU_ARE_NOT_ALLOWED_TO_CREATE_A_ROLE: import("better-auth").RawError<"YOU_ARE_NOT_ALLOWED_TO_CREATE_A_ROLE">;
        YOU_ARE_NOT_ALLOWED_TO_UPDATE_A_ROLE: import("better-auth").RawError<"YOU_ARE_NOT_ALLOWED_TO_UPDATE_A_ROLE">;
        YOU_ARE_NOT_ALLOWED_TO_DELETE_A_ROLE: import("better-auth").RawError<"YOU_ARE_NOT_ALLOWED_TO_DELETE_A_ROLE">;
        YOU_ARE_NOT_ALLOWED_TO_READ_A_ROLE: import("better-auth").RawError<"YOU_ARE_NOT_ALLOWED_TO_READ_A_ROLE">;
        YOU_ARE_NOT_ALLOWED_TO_LIST_A_ROLE: import("better-auth").RawError<"YOU_ARE_NOT_ALLOWED_TO_LIST_A_ROLE">;
        YOU_ARE_NOT_ALLOWED_TO_GET_A_ROLE: import("better-auth").RawError<"YOU_ARE_NOT_ALLOWED_TO_GET_A_ROLE">;
        TOO_MANY_ROLES: import("better-auth").RawError<"TOO_MANY_ROLES">;
        INVALID_RESOURCE: import("better-auth").RawError<"INVALID_RESOURCE">;
        ROLE_NAME_IS_ALREADY_TAKEN: import("better-auth").RawError<"ROLE_NAME_IS_ALREADY_TAKEN">;
        CANNOT_DELETE_A_PRE_DEFINED_ROLE: import("better-auth").RawError<"CANNOT_DELETE_A_PRE_DEFINED_ROLE">;
        ROLE_IS_ASSIGNED_TO_MEMBERS: import("better-auth").RawError<"ROLE_IS_ASSIGNED_TO_MEMBERS">;
    };
} | {
    id: "api-key";
    $InferServerPlugin: ReturnType<typeof import("@better-auth/api-key").apiKey>;
    pathMethods: {
        "/api-key/create": "POST";
        "/api-key/delete": "POST";
        "/api-key/delete-all-expired-api-keys": "POST";
    };
    $ERROR_CODES: {
        INVALID_METADATA_TYPE: import("better-auth").RawError<"INVALID_METADATA_TYPE">;
        REFILL_AMOUNT_AND_INTERVAL_REQUIRED: import("better-auth").RawError<"REFILL_AMOUNT_AND_INTERVAL_REQUIRED">;
        REFILL_INTERVAL_AND_AMOUNT_REQUIRED: import("better-auth").RawError<"REFILL_INTERVAL_AND_AMOUNT_REQUIRED">;
        USER_BANNED: import("better-auth").RawError<"USER_BANNED">;
        UNAUTHORIZED_SESSION: import("better-auth").RawError<"UNAUTHORIZED_SESSION">;
        KEY_NOT_FOUND: import("better-auth").RawError<"KEY_NOT_FOUND">;
        KEY_DISABLED: import("better-auth").RawError<"KEY_DISABLED">;
        KEY_EXPIRED: import("better-auth").RawError<"KEY_EXPIRED">;
        USAGE_EXCEEDED: import("better-auth").RawError<"USAGE_EXCEEDED">;
        KEY_NOT_RECOVERABLE: import("better-auth").RawError<"KEY_NOT_RECOVERABLE">;
        EXPIRES_IN_IS_TOO_SMALL: import("better-auth").RawError<"EXPIRES_IN_IS_TOO_SMALL">;
        EXPIRES_IN_IS_TOO_LARGE: import("better-auth").RawError<"EXPIRES_IN_IS_TOO_LARGE">;
        INVALID_REMAINING: import("better-auth").RawError<"INVALID_REMAINING">;
        INVALID_PREFIX_LENGTH: import("better-auth").RawError<"INVALID_PREFIX_LENGTH">;
        INVALID_NAME_LENGTH: import("better-auth").RawError<"INVALID_NAME_LENGTH">;
        METADATA_DISABLED: import("better-auth").RawError<"METADATA_DISABLED">;
        RATE_LIMIT_EXCEEDED: import("better-auth").RawError<"RATE_LIMIT_EXCEEDED">;
        NO_VALUES_TO_UPDATE: import("better-auth").RawError<"NO_VALUES_TO_UPDATE">;
        KEY_DISABLED_EXPIRATION: import("better-auth").RawError<"KEY_DISABLED_EXPIRATION">;
        INVALID_API_KEY: import("better-auth").RawError<"INVALID_API_KEY">;
        INVALID_USER_ID_FROM_API_KEY: import("better-auth").RawError<"INVALID_USER_ID_FROM_API_KEY">;
        INVALID_REFERENCE_ID_FROM_API_KEY: import("better-auth").RawError<"INVALID_REFERENCE_ID_FROM_API_KEY">;
        INVALID_API_KEY_GETTER_RETURN_TYPE: import("better-auth").RawError<"INVALID_API_KEY_GETTER_RETURN_TYPE">;
        SERVER_ONLY_PROPERTY: import("better-auth").RawError<"SERVER_ONLY_PROPERTY">;
        FAILED_TO_UPDATE_API_KEY: import("better-auth").RawError<"FAILED_TO_UPDATE_API_KEY">;
        NAME_REQUIRED: import("better-auth").RawError<"NAME_REQUIRED">;
        ORGANIZATION_ID_REQUIRED: import("better-auth").RawError<"ORGANIZATION_ID_REQUIRED">;
        USER_NOT_MEMBER_OF_ORGANIZATION: import("better-auth").RawError<"USER_NOT_MEMBER_OF_ORGANIZATION">;
        INSUFFICIENT_API_KEY_PERMISSIONS: import("better-auth").RawError<"INSUFFICIENT_API_KEY_PERMISSIONS">;
        NO_DEFAULT_API_KEY_CONFIGURATION_FOUND: import("better-auth").RawError<"NO_DEFAULT_API_KEY_CONFIGURATION_FOUND">;
        ORGANIZATION_PLUGIN_REQUIRED: import("better-auth").RawError<"ORGANIZATION_PLUGIN_REQUIRED">;
    };
})[];
/**
 * Create a pre-configured Auth client.
 *
 * This factory provides sensible defaults for Agentuity projects:
 * - Uses `/api/auth` as the default base path
 * - Automatically uses `window.location.origin` as base URL in browsers
 * - Includes organization and API key plugins by default
 *
 * @example Basic usage (zero config)
 * ```typescript
 * import { createAuthClient } from '@agentuity/auth/react';
 *
 * export const authClient = createAuthClient();
 * export const { signIn, signUp, signOut, useSession, getSession } = authClient;
 * ```
 *
 * @example With custom base path
 * ```typescript
 * export const authClient = createAuthClient({
 *   basePath: '/auth', // If mounted at /auth instead of /api/auth
 * });
 * ```
 *
 * @example With additional plugins
 * ```typescript
 * import { twoFactorClient } from 'better-auth/client/plugins';
 *
 * export const authClient = createAuthClient({
 *   plugins: [twoFactorClient()],
 * });
 * ```
 *
 * @example With custom plugins only (no defaults)
 * ```typescript
 * import { organizationClient } from 'better-auth/client/plugins';
 *
 * export const authClient = createAuthClient({
 *   skipDefaultPlugins: true,
 *   plugins: [organizationClient()],
 * });
 * ```
 */
export declare function createAuthClient<TPlugins extends BetterAuthClientPlugin[] = []>(options?: AuthClientOptions<TPlugins>): ReturnType<typeof createBetterAuthClient<{
    plugins: TPlugins;
}>>;
/**
 * Type helper for the auth client return type.
 */
export type AuthClient = ReturnType<typeof createAuthClient>;
/**
 * Context value for Auth.
 */
export interface AuthContextValue {
    /** The auth client instance */
    authClient: AuthClient;
    /** Current authenticated user, or null if not signed in */
    user: AuthUser | null;
    /** Current session object (if available) */
    session: AuthSession | null;
    /** Whether the auth state is still loading */
    isPending: boolean;
    /** Any error that occurred while fetching auth state */
    error: Error | null;
    /** Whether the user is authenticated */
    isAuthenticated: boolean;
}
export interface AuthProviderProps {
    /** React children to render */
    children: React.ReactNode;
    /**
     * The auth client instance created with createAuthClient().
     * Required for session management.
     */
    authClient: AuthClient;
    /**
     * Token refresh interval in milliseconds.
     *
     * **Default:** `3600000` (1 hour)
     *
     * Controls how frequently the auth state is refreshed by polling the session endpoint.
     * A longer interval reduces server load and API calls, but means auth state changes
     * (like session expiration or revocation) may not be detected for up to the interval duration.
     *
     * **Security Implications:**
     * - Longer intervals mean staler auth state: revoked sessions or permission changes
     *   may not be detected until the next refresh cycle (up to the interval duration)
     * - Shorter intervals provide fresher state but increase server load and API calls
     * - Consider your security requirements when choosing an interval
     *
     * **Recommended Intervals:**
     * - `30000` - `60000` (30s - 1m): High-security applications requiring near-real-time
     *   detection of session revocation or permission changes
     * - `300000` - `900000` (5m - 15m): Sensitive features (admin panels, financial operations)
     *   where timely detection of auth changes is important
     * - `3600000` (1h): Typical applications where occasional staleness is acceptable
     *
     * **Override Example:**
     * ```tsx
     * // High-security: refresh every 30 seconds
     * <AuthProvider authClient={authClient} refreshInterval={30000}>
     *
     * // Sensitive features: refresh every 5 minutes
     * <AuthProvider authClient={authClient} refreshInterval={300000}>
     *
     * // Default: refresh every hour
     * <AuthProvider authClient={authClient}>
     * ```
     */
    refreshInterval?: number;
    /**
     * Override the token endpoint path.
     * Defaults to '/token' (relative to the auth client's basePath).
     * Set to `false` to disable token fetching entirely.
     *
     * @example Custom token endpoint
     * ```tsx
     * <AuthProvider authClient={authClient} tokenEndpoint="/api/custom/jwt">
     * ```
     *
     * @example Disable token fetching
     * ```tsx
     * <AuthProvider authClient={authClient} tokenEndpoint={false}>
     * ```
     */
    tokenEndpoint?: string | false;
}
/**
 * Auth provider component.
 *
 * This component integrates Auth with Agentuity's React context,
 * automatically injecting auth tokens into API calls via useAgent and useWebsocket.
 *
 * Must be a child of AgentuityProvider.
 *
 * @example
 * ```tsx
 * import { AgentuityProvider } from '@agentuity/react';
 * import { createAuthClient, AuthProvider } from '@agentuity/auth/react';
 *
 * const authClient = createAuthClient();
 *
 * <AgentuityProvider>
 *   <AuthProvider authClient={authClient}>
 *     <App />
 *   </AuthProvider>
 * </AgentuityProvider>
 * ```
 */
export declare function AuthProvider({ children, authClient, refreshInterval, tokenEndpoint, }: AuthProviderProps): import("react/jsx-runtime").JSX.Element;
/**
 * Hook to access Auth state.
 *
 * This hook provides access to the current user and session.
 * Must be used within an AuthProvider.
 *
 * @example
 * ```tsx
 * import { useAuth } from '@agentuity/auth/react';
 *
 * function Profile() {
 *   const { user, session, isPending, isAuthenticated } = useAuth();
 *
 *   if (isPending) return <div>Loading...</div>;
 *   if (!isAuthenticated) return <div>Not signed in</div>;
 *
 *   return <div>Welcome, {user.name}!</div>;
 * }
 * ```
 */
export declare function useAuth(): AuthContextValue;
//# sourceMappingURL=react.d.ts.map