import { z } from 'zod';
import { type APIClient } from '../api.ts';
import { type CreateWebhookDestinationRequest, type UpdateWebhookDestinationRequest, type WebhookApiOptions, type WebhookDestination } from './types.ts';
export declare const WebhookDestinationResponseSchema: z.ZodDiscriminatedUnion<[z.ZodObject<{
    success: z.ZodLiteral<false>;
    message: z.ZodString;
    code: z.ZodOptional<z.ZodString>;
}, z.core.$strip>, z.ZodObject<{
    success: z.ZodLiteral<true>;
    data: z.ZodObject<{
        id: z.ZodString;
        created_at: z.ZodString;
        updated_at: z.ZodString;
        created_by: z.ZodString;
        webhook_id: z.ZodString;
        type: z.ZodEnum<{
            url: "url";
        }>;
        config: z.ZodRecord<z.ZodString, z.ZodUnknown>;
    }, z.core.$strip>;
}, z.core.$strip>], "success">;
export declare const WebhookDestinationsListResponseSchema: z.ZodDiscriminatedUnion<[z.ZodObject<{
    success: z.ZodLiteral<false>;
    message: z.ZodString;
    code: z.ZodOptional<z.ZodString>;
}, z.core.$strip>, z.ZodObject<{
    success: z.ZodLiteral<true>;
    data: z.ZodArray<z.ZodObject<{
        id: z.ZodString;
        created_at: z.ZodString;
        updated_at: z.ZodString;
        created_by: z.ZodString;
        webhook_id: z.ZodString;
        type: z.ZodEnum<{
            url: "url";
        }>;
        config: z.ZodRecord<z.ZodString, z.ZodUnknown>;
    }, z.core.$strip>>;
}, z.core.$strip>], "success">;
export declare const DeleteWebhookDestinationResponseSchema: z.ZodDiscriminatedUnion<[z.ZodObject<{
    success: z.ZodLiteral<false>;
    message: z.ZodString;
    code: z.ZodOptional<z.ZodString>;
}, z.core.$strip>, z.ZodObject<{
    success: z.ZodLiteral<true>;
}, z.core.$strip>], "success">;
/**
 * Create a destination for a webhook.
 *
 * Destinations define where incoming webhook payloads are forwarded to.
 * When a webhook receives a payload, it will be delivered to all configured destinations.
 *
 * @param client - The API client instance
 * @param webhookId - The webhook ID (prefixed with wh_)
 * @param params - Destination configuration including type and config
 * @param options - Optional API options (e.g., orgId)
 * @returns The created destination
 * @throws {WebhookNotFoundError} If the webhook does not exist
 * @throws {WebhookError} If the API request fails
 *
 * @example
 * ```typescript
 * const destination = await createWebhookDestination(client, 'wh_abc123', {
 *   type: 'url',
 *   config: { url: 'https://api.example.com/webhook' },
 * });
 * console.log(`Created destination ${destination.id}`);
 * ```
 */
export declare function createWebhookDestination(client: APIClient, webhookId: string, params: CreateWebhookDestinationRequest, options?: WebhookApiOptions): Promise<WebhookDestination>;
/**
 * List all destinations for a webhook.
 *
 * Retrieves all destinations configured for a webhook.
 *
 * @param client - The API client instance
 * @param webhookId - The webhook ID (prefixed with wh_)
 * @param options - Optional API options (e.g., orgId)
 * @returns Array of destinations configured for the webhook
 * @throws {WebhookNotFoundError} If the webhook does not exist
 * @throws {WebhookError} If the API request fails
 *
 * @example
 * ```typescript
 * const destinations = await listWebhookDestinations(client, 'wh_abc123');
 * for (const dest of destinations) {
 *   console.log(`Destination ${dest.id}: type=${dest.type}`);
 * }
 * ```
 */
export declare function listWebhookDestinations(client: APIClient, webhookId: string, options?: WebhookApiOptions): Promise<WebhookDestination[]>;
/**
 * Update a webhook destination's configuration.
 *
 * Modifies an existing destination's config. Only the fields provided in params will be updated.
 *
 * @param client - The API client instance
 * @param webhookId - The webhook ID (prefixed with wh_)
 * @param destinationId - The destination ID to update (prefixed with whds_)
 * @param params - Fields to update
 * @param options - Optional API options (e.g., orgId)
 * @returns The updated destination
 * @throws {WebhookDestinationNotFoundError} If the destination does not exist
 * @throws {WebhookNotFoundError} If the webhook does not exist
 * @throws {WebhookError} If the API request fails
 *
 * @example
 * ```typescript
 * const updated = await updateWebhookDestination(client, 'wh_abc123', 'whds_def456', {
 *   config: { url: 'https://api.example.com/webhook/v2' },
 * });
 * ```
 */
export declare function updateWebhookDestination(client: APIClient, webhookId: string, destinationId: string, params: UpdateWebhookDestinationRequest, options?: WebhookApiOptions): Promise<WebhookDestination>;
/**
 * Delete a webhook destination.
 *
 * Permanently removes a destination. Webhook payloads will no longer be
 * forwarded to this endpoint. This action cannot be undone.
 *
 * @param client - The API client instance
 * @param webhookId - The webhook ID (prefixed with wh_)
 * @param destinationId - The destination ID to delete (prefixed with whds_)
 * @param options - Optional API options (e.g., orgId)
 * @throws {WebhookDestinationNotFoundError} If the destination does not exist
 * @throws {WebhookNotFoundError} If the webhook does not exist
 * @throws {WebhookError} If the API request fails
 *
 * @example
 * ```typescript
 * await deleteWebhookDestination(client, 'wh_abc123', 'whds_def456');
 * console.log('Destination deleted');
 * ```
 */
export declare function deleteWebhookDestination(client: APIClient, webhookId: string, destinationId: string, options?: WebhookApiOptions): Promise<void>;
//# sourceMappingURL=destinations.d.ts.map