import { z } from 'zod';
import { type APIClient } from '../api.ts';
/**
 * Data schema for sandbox resolve endpoint
 */
export declare const SandboxResolveDataSchema: z.ZodObject<{
    id: z.ZodString;
    name: z.ZodNullable<z.ZodString>;
    region: z.ZodString;
    status: z.ZodString;
    orgId: z.ZodString;
    projectId: z.ZodNullable<z.ZodString>;
}, z.core.$strip>;
/**
 * Response schema for sandbox resolve endpoint using standardized discriminated union
 */
export declare const SandboxResolveResponseSchema: z.ZodDiscriminatedUnion<[z.ZodObject<{
    success: z.ZodLiteral<false>;
    message: z.ZodString;
    code: z.ZodOptional<z.ZodString>;
}, z.core.$strip>, z.ZodObject<{
    success: z.ZodLiteral<true>;
    data: z.ZodOptional<z.ZodObject<{
        id: z.ZodString;
        name: z.ZodNullable<z.ZodString>;
        region: z.ZodString;
        status: z.ZodString;
        orgId: z.ZodString;
        projectId: z.ZodNullable<z.ZodString>;
    }, z.core.$strip>>;
}, z.core.$strip>], "success">;
/**
 * Resolved sandbox info returned from the CLI API
 */
export interface ResolvedSandboxInfo {
    id: string;
    name: string | null;
    region: string;
    status: string;
    orgId: string;
    projectId: string | null;
}
/**
 * Error thrown when sandbox resolution fails.
 *
 * @example
 * ```typescript
 * try {
 *   await sandboxResolve(client, 'sbx_123');
 * } catch (error) {
 *   if (error._tag === 'SandboxResolveError') {
 *     console.error(`Sandbox not found: ${error.sandboxId}`);
 *   }
 * }
 * ```
 */
export declare const SandboxResolveError: {
    new (args?: ({
        sandboxId?: string;
        statusCode?: number;
    } & {
        message?: string;
        cause?: unknown;
    }) | undefined): import("@agentuity/core").RichError & {
        readonly _tag: "SandboxResolveError";
    } & Readonly<{
        sandboxId?: string;
        statusCode?: number;
    }>;
    readonly defaultMessage?: string;
};
/**
 * Resolve a sandbox by ID across all organizations the user has access to.
 * Uses the CLI API endpoint which searches across all user's orgs.
 *
 * @param client - API client configured for CLI endpoints
 * @param sandboxId - The sandbox ID to resolve
 * @returns Resolved sandbox info including region and orgId
 * @throws {SandboxResolveError} If sandbox not found or request fails
 */
export declare function sandboxResolve(client: APIClient, sandboxId: string): Promise<ResolvedSandboxInfo>;
//# sourceMappingURL=resolve.d.ts.map