/**
 * @module websocket
 *
 * WebSocket client for real-time queue message subscriptions.
 *
 * Provides both a callback-based API ({@link createQueueWebSocket}) and an
 * async iterator API ({@link subscribeToQueue}) for receiving messages from
 * a queue in real time via WebSocket.
 *
 * @example Callback-based API
 * ```typescript
 * import { createQueueWebSocket } from '@agentuity/server';
 *
 * const connection = createQueueWebSocket({
 *     queueName: 'order-processing',
 *     baseUrl: 'https://catalyst.agentuity.cloud',
 *     onMessage: (message) => {
 *         console.log('Received:', message.id, message.payload);
 *     },
 *     onOpen: () => console.log('Connected'),
 *     onClose: (code, reason) => console.log('Closed:', code, reason),
 *     onError: (error) => console.error('Error:', error),
 * });
 *
 * // Later: close the connection
 * connection.close();
 * ```
 *
 * @example Resuming from a previous session
 * ```typescript
 * import { createQueueWebSocket } from '@agentuity/server';
 *
 * // Use a previously obtained clientId and lastOffset to resume
 * const connection = createQueueWebSocket({
 *     queueName: 'order-processing',
 *     baseUrl: 'https://catalyst.agentuity.cloud',
 *     clientId: previousClientId,
 *     lastOffset: previousOffset,
 *     onMessage: (message) => {
 *         console.log('Received:', message.id, message.payload);
 *     },
 * });
 * ```
 *
 * @example Async iterator API
 * ```typescript
 * import { subscribeToQueue } from '@agentuity/server';
 *
 * const controller = new AbortController();
 * for await (const message of subscribeToQueue({
 *     queueName: 'order-processing',
 *     baseUrl: 'https://catalyst.agentuity.cloud',
 *     signal: controller.signal,
 * })) {
 *     console.log('Received:', message.id, message.payload);
 * }
 * ```
 */
import type { Message } from './types.ts';
/** Connection state for a queue WebSocket connection. */
export type QueueWebSocketState = 'connecting' | 'authenticating' | 'connected' | 'reconnecting' | 'closed';
/** Options for creating a queue WebSocket subscription. */
export interface QueueWebSocketOptions {
    /** Queue name to subscribe to. */
    queueName: string;
    /** API key for authentication (if not provided, uses AGENTUITY_SDK_KEY env var). */
    apiKey?: string;
    /** Base URL of the catalyst service (e.g., https://catalyst.agentuity.cloud). */
    baseUrl: string;
    /** Called when a message is received. */
    onMessage: (message: Message) => void;
    /** Called when the connection is established and authenticated. */
    onOpen?: () => void;
    /** Called when the connection is closed. */
    onClose?: (code: number, reason: string) => void;
    /** Called when an error occurs. */
    onError?: (error: Error) => void;
    /** Whether to automatically reconnect on disconnection (default: true). */
    autoReconnect?: boolean;
    /** Maximum number of reconnection attempts (default: Infinity). */
    maxReconnectAttempts?: number;
    /** Initial reconnection delay in ms (default: 1000). Uses exponential backoff. */
    reconnectDelayMs?: number;
    /** Maximum reconnection delay in ms (default: 30000). */
    maxReconnectDelayMs?: number;
    /** Optional client ID from a previous session to resume a subscription. */
    clientId?: string;
    /** Optional last processed offset from a previous session to resume from. */
    lastOffset?: number;
}
/** Return type from {@link createQueueWebSocket}. */
export interface QueueWebSocketConnection {
    /** Close the WebSocket connection. Disables auto-reconnect. */
    close(): void;
    /** The current connection state. */
    readonly state: QueueWebSocketState;
    /** The client/subscription ID assigned by the server. Stable across reconnections. */
    readonly clientId: string | undefined;
    /** The offset of the last message processed. */
    readonly lastOffset: number | undefined;
}
/** Options for the async iterator queue subscription. */
export interface SubscribeToQueueOptions {
    /** Queue name to subscribe to. */
    queueName: string;
    /** API key for authentication (if not provided, uses AGENTUITY_SDK_KEY env var). */
    apiKey?: string;
    /** Base URL of the catalyst service. */
    baseUrl: string;
    /** AbortSignal to stop the subscription. */
    signal?: AbortSignal;
    /** Optional client ID from a previous session to resume a subscription. */
    clientId?: string;
    /** Optional last processed offset from a previous session to resume from. */
    lastOffset?: number;
}
/**
 * Create a WebSocket connection to receive real-time messages from a queue.
 *
 * The connection handles authentication, automatic reconnection with exponential
 * backoff, and ping/pong keep-alive (handled automatically by the WebSocket
 * implementation).
 *
 * @param options - Configuration for the WebSocket connection
 * @returns A {@link QueueWebSocketConnection} handle for managing the connection
 * @throws {QueueError} If no API key is available
 * @throws {QueueValidationError} If the queue name is invalid
 *
 * @example
 * ```typescript
 * const connection = createQueueWebSocket({
 *     queueName: 'order-processing',
 *     baseUrl: 'https://catalyst.agentuity.cloud',
 *     onMessage: (message) => {
 *         console.log('Received:', message.id, message.payload);
 *     },
 * });
 *
 * // Later: close the connection
 * connection.close();
 * ```
 */
export declare function createQueueWebSocket(options: QueueWebSocketOptions): QueueWebSocketConnection;
/**
 * Subscribe to real-time messages from a queue via WebSocket.
 *
 * Returns an async iterator that yields messages as they arrive.
 * The connection is automatically managed (auth, reconnection, cleanup).
 *
 * @param options - Configuration for the subscription
 * @returns An async generator that yields {@link Message} objects
 * @throws {QueueError} If no API key is available
 * @throws {QueueValidationError} If the queue name is invalid
 *
 * @example
 * ```typescript
 * const controller = new AbortController();
 * for await (const message of subscribeToQueue({
 *     queueName: 'order-processing',
 *     baseUrl: 'https://catalyst.agentuity.cloud',
 *     signal: controller.signal,
 * })) {
 *     console.log('Received:', message.id, message.payload);
 * }
 * ```
 */
export declare function subscribeToQueue(options: SubscribeToQueueOptions): AsyncGenerator<Message, void, unknown>;
//# sourceMappingURL=websocket.d.ts.map