import { z } from 'zod';
import { APIError, APIResponseSchema } from "../api.js";
import { ProjectResponseError } from "./util.js";
export const ProjectExistsRequestSchema = z.object({
    name: z.string().max(255).min(1).describe('the name of the new project'),
    organization_id: z
        .string()
        .max(255)
        .min(1)
        .describe('the organization id to create the project in'),
});
export const ProjectExistsResponseSchema = APIResponseSchema(z.boolean());
/**
 * Check if a project exists by name within an organization
 *
 * @param client
 * @param body
 * @returns
 */
export async function projectExists(client, body) {
    const qs = new URLSearchParams();
    qs.set('orgId', body.organization_id);
    try {
        const resp = await client.request('GET', `/cli/project/exists/${encodeURIComponent(body.name)}?${qs.toString()}`, ProjectExistsResponseSchema);
        if (resp.success) {
            return resp.data;
        }
        return false;
    }
    catch (ex) {
        const _ex = ex;
        if (_ex instanceof APIError) {
            switch (_ex.status) {
                case 409:
                    return true;
                case 422:
                    return false;
                default:
                    break;
            }
        }
        throw new ProjectResponseError({ cause: ex });
    }
}
//# sourceMappingURL=exists.js.map