export { createIssue, success, failure, ValidationError, SCHEMA_KIND } from "./base.js";
export { StringSchema, string } from "./primitives/string.js";
export { NumberSchema, number } from "./primitives/number.js";
export { BooleanSchema, boolean } from "./primitives/boolean.js";
export { NullSchema, null_ } from "./primitives/null.js";
export { UndefinedSchema, undefined_ } from "./primitives/undefined.js";
export { UnknownSchema, unknown } from "./primitives/unknown.js";
export { AnySchema, any } from "./primitives/any.js";
export { ObjectSchema, object } from "./complex/object.js";
export { ArraySchema, array } from "./complex/array.js";
export { RecordSchema, record } from "./complex/record.js";
export { LiteralSchema, literal } from "./utils/literal.js";
export { OptionalSchema, optional } from "./utils/optional.js";
export { NullableSchema, nullable } from "./utils/nullable.js";
export { UnionSchema, union } from "./utils/union.js";
export { toJSONSchema, fromJSONSchema, } from "./json-schema.js";
export { CoerceStringSchema, coerceString } from "./coerce/string.js";
export { CoerceNumberSchema, coerceNumber } from "./coerce/number.js";
export { CoerceBooleanSchema, coerceBoolean } from "./coerce/boolean.js";
export { CoerceDateSchema, coerceDate } from "./coerce/date.js";
import { string } from "./primitives/string.js";
import { number } from "./primitives/number.js";
import { boolean } from "./primitives/boolean.js";
import { null_ } from "./primitives/null.js";
import { undefined_ } from "./primitives/undefined.js";
import { unknown } from "./primitives/unknown.js";
import { any } from "./primitives/any.js";
import { object } from "./complex/object.js";
import { array } from "./complex/array.js";
import { record } from "./complex/record.js";
import { literal } from "./utils/literal.js";
import { optional } from "./utils/optional.js";
import { nullable } from "./utils/nullable.js";
import { union } from "./utils/union.js";
import { toJSONSchema, fromJSONSchema } from "./json-schema.js";
import { coerceString } from "./coerce/string.js";
import { coerceNumber } from "./coerce/number.js";
import { coerceBoolean } from "./coerce/boolean.js";
import { coerceDate } from "./coerce/date.js";
/**
 * Create an enum schema (union of literal values).
 * Shorthand for creating a union of literals.
 *
 * @param values - Array of literal values
 *
 * @example
 * ```typescript
 * const roleSchema = s.enum(['admin', 'user', 'guest']);
 * const role = roleSchema.parse('admin'); // 'admin'
 *
 * // Equivalent to:
 * s.union(s.literal('admin'), s.literal('user'), s.literal('guest'))
 * ```
 */
function enumSchema(values) {
    return union(...values.map((v) => literal(v)));
}
/**
 * Main schema builder object.
 * Provides access to all schema types and utilities.
 *
 * @example
 * ```typescript
 * import { s } from '@agentuity/schema';
 *
 * // Define a schema
 * const User = s.object({
 *   name: s.string(),
 *   age: s.number(),
 *   role: s.enum(['admin', 'user'])
 * });
 *
 * // Extract type
 * type User = s.infer<typeof User>;
 *
 * // Parse data
 * const user = User.parse(data);
 * ```
 */
export const s = {
    /** Create a string schema */
    string,
    /** Create a number schema */
    number,
    /** Create a boolean schema */
    boolean,
    /** Create a null schema */
    null: null_,
    /** Create an undefined schema */
    undefined: undefined_,
    /** Create an unknown schema (accepts any value) */
    unknown,
    /** Create an any schema (accepts any value) */
    any,
    /** Create an object schema with typed properties */
    object,
    /** Create an array schema with typed elements */
    array,
    /** Create a record schema (object with string keys and typed values) */
    record,
    /** Create a literal value schema */
    literal,
    /** Make a schema optional (T | undefined) */
    optional,
    /** Make a schema nullable (T | null) */
    nullable,
    /** Create a union of schemas */
    union,
    /** Create an enum schema (union of literals) */
    enum: enumSchema,
    /** Convert schema to JSON Schema format */
    toJSONSchema,
    /** Convert JSON Schema to schema */
    fromJSONSchema,
    /** Coercion schemas for type conversion */
    coerce: {
        /** Coerce to string using String(value) */
        string: coerceString,
        /** Coerce to number using Number(value) */
        number: coerceNumber,
        /** Coerce to boolean using Boolean(value) */
        boolean: coerceBoolean,
        /** Coerce to Date using new Date(value) */
        date: coerceDate,
    },
};
//# sourceMappingURL=index.js.map