import type { Context, Handler } from 'hono';
import type { Env } from '../app';
/**
 * Handler function for streaming responses.
 * Returns a ReadableStream that will be piped to the response.
 */
export type StreamHandler<E extends Env = Env> = (c: Context<E>) => ReadableStream<Uint8Array | string> | Promise<ReadableStream<Uint8Array | string>>;
/**
 * Creates a streaming middleware for returning ReadableStream responses.
 *
 * Use with router.post() (or any HTTP method) to create a streaming endpoint:
 *
 * @example
 * ```typescript
 * import { createRouter, stream } from '@agentuity/runtime';
 *
 * const router = createRouter();
 *
 * router.post('/events', stream((c) => {
 *   return new ReadableStream({
 *     start(controller) {
 *       controller.enqueue('event 1\n');
 *       controller.enqueue('event 2\n');
 *       controller.close();
 *     }
 *   });
 * }));
 * ```
 *
 * @example
 * ```typescript
 * // Async stream with data from request body
 * router.post('/process', stream(async (c) => {
 *   const body = await c.req.json();
 *
 *   return new ReadableStream({
 *     async start(controller) {
 *       for (const item of body.items) {
 *         controller.enqueue(`Processing: ${item}\n`);
 *         await new Promise(r => setTimeout(r, 100));
 *       }
 *       controller.close();
 *     }
 *   });
 * }));
 * ```
 *
 * @param handler - Handler function returning a ReadableStream
 * @returns Hono handler for streaming response
 */
export declare function stream<E extends Env = Env>(handler: StreamHandler<E>): Handler<E>;
//# sourceMappingURL=stream.d.ts.map