import { type Context } from '@opentelemetry/api';
import type { KeyValueStorage, StreamStorage, VectorStorage, SandboxService, QueueService, EmailService, ScheduleService, TaskStorage, Logger } from '@agentuity/core';
import type { AgentContext, AgentRegistry, AgentRuntimeState, AgentMetadata, AgentRunner } from './agent';
import { AGENT_RUNTIME, AGENT_IDS } from './_config';
import type { Thread, Session } from './session';
/**
 * Options for creating a standalone agent context.
 *
 * Use this when executing agents outside of HTTP requests (Discord bots, cron jobs, etc.)
 */
export interface StandaloneContextOptions {
    /**
     * Session ID for this execution. If not provided, will be auto-generated from trace context.
     */
    sessionId?: string;
    /**
     * Thread for multi-turn conversations. If not provided, will be restored/created from thread provider.
     */
    thread?: Thread;
    /**
     * Session for this execution. If not provided, will be created.
     */
    session?: Session;
    /**
     * Parent OpenTelemetry context for distributed tracing.
     */
    parentContext?: Context;
    /**
     * Trigger type for this execution (used in telemetry and session events).
     */
    trigger?: import('@agentuity/core').SessionStartEvent['trigger'];
}
/**
 * Options for invoke() method.
 */
export interface InvokeOptions {
    /**
     * Span name for OpenTelemetry trace (default: 'agent-invocation')
     */
    spanName?: string;
}
/**
 * Standalone agent context for executing agents outside of HTTP requests.
 *
 * This context provides the same infrastructure as HTTP request contexts:
 * - OpenTelemetry tracing with proper span hierarchy
 * - Session and thread management (save/restore)
 * - Background task handling (waitUntil)
 * - Session event tracking (start/complete)
 * - Access to all services (kv, stream, vector)
 *
 * @example
 * ```typescript
 * import { createAgentContext } from '@agentuity/runtime';
 * import myAgent from './agents/my-agent';
 *
 * // Simple usage:
 * const ctx = createAgentContext();
 * const result = await ctx.invoke(() => myAgent.run(input));
 *
 * // With custom session tracking:
 * const ctx = createAgentContext({
 *   sessionId: discordMessage.id,
 *   trigger: 'discord'
 * });
 * const result = await ctx.invoke(() => myAgent.run(input));
 *
 * // Reuse context for multiple agents:
 * const ctx = createAgentContext();
 * const result1 = await ctx.invoke(() => agent1.run(input1));
 * const result2 = await ctx.invoke(() => agent2.run(result1));
 * ```
 */
export declare class StandaloneAgentContext<TAgentMap extends AgentRegistry = AgentRegistry, TConfig = unknown, TAppState = Record<string, never>> implements AgentContext<TAgentMap, TConfig, TAppState> {
    agent: TAgentMap;
    logger: Logger;
    tracer: import('@opentelemetry/api').Tracer;
    kv: KeyValueStorage;
    stream: StreamStorage;
    vector: VectorStorage;
    sandbox: SandboxService;
    queue: QueueService;
    email: EmailService;
    schedule: ScheduleService;
    task: TaskStorage;
    config: TConfig;
    app: TAppState;
    current: AgentMetadata;
    [AGENT_RUNTIME]: AgentRuntimeState;
    sessionId: string;
    state: Map<string, unknown>;
    session: Session;
    thread: Thread;
    auth: import('@agentuity/auth/types').AuthInterface | null;
    [AGENT_IDS]?: Set<string>;
    private readonly parentContext;
    private readonly trigger;
    private readonly initialSessionId?;
    constructor(options?: StandaloneContextOptions);
    waitUntil(_callback: Promise<void> | (() => void | Promise<void>)): void;
    /**
     * Execute a function within this agent context.
     *
     * This method:
     * 1. Creates an OpenTelemetry span for the invocation
     * 2. Restores/creates session and thread
     * 3. Sends session start event
     * 4. Executes the function within AsyncLocalStorage context
     * 5. Waits for background tasks (waitUntil)
     * 6. Saves session and thread
     * 7. Sends session complete event
     *
     * @param fn - Function to execute (typically () => agent.run(input))
     * @param options - Optional configuration for the invocation
     * @returns Promise that resolves to the function's return value
     *
     * @example
     * ```typescript
     * const result = await ctx.invoke(() => myAgent.run({ userId: '123' }));
     * ```
     *
     * @example
     * ```typescript
     * // Multiple agents in sequence:
     * const result = await ctx.invoke(async () => {
     *   const step1 = await agent1.run(input);
     *   return agent2.run(step1);
     * });
     * ```
     */
    invoke<T>(fn: () => Promise<T>, options?: InvokeOptions): Promise<T>;
    /**
     * Execute an agent with the given input within this context.
     *
     * This is a convenience method that wraps `invoke()` for cleaner syntax
     * when running a single agent.
     *
     * @param agent - The agent to execute (must have a `run` method)
     * @param input - Input to pass to the agent (if agent requires input)
     * @returns Promise that resolves to the agent's output
     *
     * @example
     * ```typescript
     * import { createAgentContext } from '@agentuity/runtime';
     * import myAgent from './agents/my-agent';
     *
     * const ctx = createAgentContext();
     * const result = await ctx.run(myAgent, { name: 'World' });
     * ```
     *
     * @example
     * ```typescript
     * // Agent without input
     * const result = await ctx.run(statusAgent);
     * ```
     *
     * @example
     * ```typescript
     * // Multiple agents in sequence
     * const ctx = createAgentContext();
     * const step1 = await ctx.run(preprocessAgent, rawData);
     * const step2 = await ctx.run(processAgent, step1);
     * const result = await ctx.run(postprocessAgent, step2);
     * ```
     */
    run<TOutput>(agent: AgentRunner<any, any, any>, input?: unknown): Promise<TOutput>;
}
/**
 * Create a standalone agent context for executing agents outside of HTTP requests.
 *
 * This is useful for Discord bots, cron jobs, CLI scripts, sandboxes, or any scenario
 * where you need to run agents but don't have an HTTP request context.
 *
 * **Auto-initialization**: When running outside of the Agentuity runtime (dev server or cloud),
 * this function automatically initializes minimal runtime globals. No manual setup required.
 *
 * @param options - Optional configuration for the context
 * @returns A StandaloneAgentContext instance
 *
 * @example
 * ```typescript
 * import { createAgentContext } from '@agentuity/runtime';
 * import myAgent from './agents/my-agent';
 *
 * // Simple usage with ctx.run() (recommended):
 * const ctx = createAgentContext();
 * const result = await ctx.run(myAgent, { name: 'World' });
 *
 * // Discord bot example:
 * client.on('messageCreate', async (message) => {
 *   const ctx = createAgentContext({
 *     sessionId: message.id,
 *     trigger: 'discord'
 *   });
 *   const response = await ctx.run(chatAgent, { message: message.content });
 *   await message.reply(response.text);
 * });
 *
 * // Cron job example:
 * cron.schedule('0 * * * *', async () => {
 *   const ctx = createAgentContext({ trigger: 'cron' });
 *   await ctx.run(cleanupAgent);
 * });
 *
 * // Multiple agents in sequence:
 * const ctx = createAgentContext();
 * const step1 = await ctx.run(preprocessAgent, rawData);
 * const result = await ctx.run(processAgent, step1);
 * ```
 */
export declare function createAgentContext<TAppState = Record<string, never>>(options?: StandaloneContextOptions): StandaloneAgentContext<AgentRegistry, unknown, TAppState>;
//# sourceMappingURL=_standalone.d.ts.map