import type { SQL, SQLQuery } from 'bun';
/**
 * Represents a PostgreSQL transaction with support for savepoints.
 *
 * Transactions are created via `PostgresClient.begin()` and support
 * tagged template literal syntax for queries.
 */
export declare class Transaction {
    private _sql;
    private _connection;
    private _committed;
    private _rolledBack;
    private _savepointCounter;
    constructor(sql: SQL, connection: SQLQuery);
    /**
     * Whether the transaction has been committed.
     */
    get committed(): boolean;
    /**
     * Whether the transaction has been rolled back.
     */
    get rolledBack(): boolean;
    /**
     * Whether the transaction is still active (not committed or rolled back).
     */
    get active(): boolean;
    /**
     * Execute a query within this transaction using tagged template literal syntax.
     *
     * @example
     * ```typescript
     * const tx = await client.begin();
     * const result = await tx`SELECT * FROM users WHERE id = ${userId}`;
     * await tx.commit();
     * ```
     */
    query(strings: TemplateStringsArray, ...values: unknown[]): SQLQuery;
    /**
     * Create a savepoint within this transaction.
     *
     * @param name - Optional name for the savepoint. If not provided, a unique name is generated.
     *               If provided, must be a valid SQL identifier (alphanumeric and underscores only,
     *               starting with a letter or underscore).
     * @returns A Savepoint object that can be used to rollback to this point.
     * @throws {TransactionError} If the provided name is not a valid SQL identifier.
     *
     * @example
     * ```typescript
     * const tx = await client.begin();
     * await tx`INSERT INTO users (name) VALUES ('Alice')`;
     *
     * const sp = await tx.savepoint();
     * await tx`INSERT INTO users (name) VALUES ('Bob')`;
     *
     * // Oops, rollback Bob but keep Alice
     * await sp.rollback();
     *
     * await tx.commit(); // Only Alice is committed
     * ```
     */
    savepoint(name?: string): Promise<Savepoint>;
    /**
     * Commit the transaction.
     *
     * @throws {TransactionError} If the transaction is not active or commit fails.
     */
    commit(): Promise<void>;
    /**
     * Rollback the transaction.
     *
     * @throws {TransactionError} If the transaction is not active or rollback fails.
     */
    rollback(): Promise<void>;
    /**
     * Releases the underlying reserved connection back to the pool.
     * Called automatically on commit or rollback. Safe to call multiple times.
     */
    private _releaseConnection;
    /**
     * Ensures the transaction is still active.
     */
    private _ensureActive;
}
/**
 * Represents a savepoint within a transaction.
 */
export declare class Savepoint {
    private _sql;
    private _name;
    private _released;
    private _rolledBack;
    constructor(sql: SQL, name: string);
    /**
     * The name of this savepoint.
     */
    get name(): string;
    /**
     * Whether the savepoint has been released.
     */
    get released(): boolean;
    /**
     * Whether the savepoint has been rolled back to.
     */
    get rolledBack(): boolean;
    /**
     * Rollback to this savepoint.
     * All changes made after this savepoint was created will be undone.
     *
     * @throws {TransactionError} If the savepoint has been released or already rolled back.
     */
    rollback(): Promise<void>;
    /**
     * Release this savepoint.
     * The savepoint is destroyed but changes are kept.
     */
    release(): Promise<void>;
}
/**
 * Represents a reserved (exclusive) connection from the pool.
 *
 * Reserved connections are created via `PostgresClient.reserve()` and support
 * tagged template literal syntax for queries.
 */
export declare class ReservedConnection {
    private _sql;
    private _released;
    constructor(sql: SQL);
    /**
     * Whether the connection has been released back to the pool.
     */
    get released(): boolean;
    /**
     * Execute a query on this reserved connection using tagged template literal syntax.
     *
     * @example
     * ```typescript
     * const conn = await client.reserve();
     * try {
     *   await conn`SET LOCAL timezone = 'UTC'`;
     *   const result = await conn`SELECT NOW()`;
     * } finally {
     *   conn.release();
     * }
     * ```
     */
    query(strings: TemplateStringsArray, ...values: unknown[]): SQLQuery;
    /**
     * Release the connection back to the pool.
     */
    release(): void;
}
//# sourceMappingURL=transaction.d.ts.map