import { buildUrl, toServiceException } from "./_util.js";
import { safeStringify } from "../json.js";
/**
 * Unwrap a Catalyst API response payload.
 * Handles both `{ key: data }` and `{ data: { key: data } }` response formats.
 */
function unwrap(payload, key) {
    if (typeof payload === 'object' && payload !== null) {
        const obj = payload;
        if (key in obj) {
            return obj[key];
        }
        if ('data' in obj && typeof obj.data === 'object' && obj.data !== null) {
            const data = obj.data;
            if (key in data) {
                return data[key];
            }
            return data;
        }
    }
    return payload;
}
export class EmailStorageService {
    #adapter;
    #baseUrl;
    constructor(baseUrl, adapter) {
        this.#adapter = adapter;
        this.#baseUrl = baseUrl;
    }
    async createAddress(localPart) {
        const url = buildUrl(this.#baseUrl, '/email/2025-03-17/addresses');
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'POST',
            body: safeStringify({ local_part: localPart }),
            contentType: 'application/json',
            signal,
            telemetry: {
                name: 'agentuity.email.createAddress',
                attributes: {
                    localPart,
                },
            },
        });
        if (res.ok) {
            return unwrap(res.data, 'address');
        }
        throw await toServiceException('POST', url, res.response);
    }
    async listAddresses() {
        const url = buildUrl(this.#baseUrl, '/email/2025-03-17/addresses');
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.email.listAddresses',
                attributes: {},
            },
        });
        if (res.response.status === 404) {
            return [];
        }
        if (res.ok) {
            const items = unwrap(res.data, 'addresses');
            return Array.isArray(items) ? items : [];
        }
        throw await toServiceException('GET', url, res.response);
    }
    async getAddress(id) {
        const url = buildUrl(this.#baseUrl, `/email/2025-03-17/addresses/${encodeURIComponent(id)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.email.getAddress',
                attributes: {
                    id,
                },
            },
        });
        if (res.response.status === 404) {
            return null;
        }
        if (res.ok) {
            return unwrap(res.data, 'address');
        }
        throw await toServiceException('GET', url, res.response);
    }
    async deleteAddress(id) {
        const url = buildUrl(this.#baseUrl, `/email/2025-03-17/addresses/${encodeURIComponent(id)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'DELETE',
            signal,
            telemetry: {
                name: 'agentuity.email.deleteAddress',
                attributes: {
                    id,
                },
            },
        });
        if (res.ok || res.response.status === 404) {
            return;
        }
        throw await toServiceException('DELETE', url, res.response);
    }
    async createDestination(addressId, type, config) {
        const url = buildUrl(this.#baseUrl, `/email/2025-03-17/addresses/${encodeURIComponent(addressId)}/destinations`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'POST',
            body: safeStringify({ type, config }),
            contentType: 'application/json',
            signal,
            telemetry: {
                name: 'agentuity.email.createDestination',
                attributes: {
                    addressId,
                    type,
                },
            },
        });
        if (res.ok) {
            return unwrap(res.data, 'destination');
        }
        throw await toServiceException('POST', url, res.response);
    }
    async listDestinations(addressId) {
        const url = buildUrl(this.#baseUrl, `/email/2025-03-17/addresses/${encodeURIComponent(addressId)}/destinations`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.email.listDestinations',
                attributes: {
                    addressId,
                },
            },
        });
        if (res.response.status === 404) {
            return [];
        }
        if (res.ok) {
            const items = unwrap(res.data, 'destinations');
            return Array.isArray(items) ? items : [];
        }
        throw await toServiceException('GET', url, res.response);
    }
    async deleteDestination(addressId, destinationId) {
        const url = buildUrl(this.#baseUrl, `/email/2025-03-17/addresses/${encodeURIComponent(addressId)}/destinations/${encodeURIComponent(destinationId)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'DELETE',
            signal,
            telemetry: {
                name: 'agentuity.email.deleteDestination',
                attributes: {
                    addressId,
                    destinationId,
                },
            },
        });
        if (res.ok || res.response.status === 404) {
            return;
        }
        throw await toServiceException('DELETE', url, res.response);
    }
    async send(params) {
        const url = buildUrl(this.#baseUrl, '/email/2025-03-17/outbound/send');
        const signal = AbortSignal.timeout(30_000);
        // Transform attachments to API format (snake_case)
        const body = {
            from: params.from,
            to: params.to,
            subject: params.subject,
        };
        if (params.text !== undefined) {
            body.text = params.text;
        }
        if (params.html !== undefined) {
            body.html = params.html;
        }
        if (params.attachments && params.attachments.length > 0) {
            body.attachments = params.attachments.map((a) => ({
                filename: a.filename,
                content: a.content,
                ...(a.contentType && { content_type: a.contentType }),
            }));
        }
        const res = await this.#adapter.invoke(url, {
            method: 'POST',
            body: safeStringify(body),
            contentType: 'application/json',
            signal,
            telemetry: {
                name: 'agentuity.email.send',
                attributes: {
                    from: params.from,
                    toCount: String(params.to.length),
                },
            },
        });
        if (res.ok) {
            return unwrap(res.data, 'outbound');
        }
        throw await toServiceException('POST', url, res.response);
    }
    async listInbound(addressId) {
        const queryParams = new URLSearchParams();
        if (addressId) {
            queryParams.set('address_id', addressId);
        }
        const queryString = queryParams.toString();
        const url = buildUrl(this.#baseUrl, `/email/2025-03-17/inbound${queryString ? `?${queryString}` : ''}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.email.listInbound',
                attributes: {
                    ...(addressId && { addressId }),
                },
            },
        });
        if (res.response.status === 404) {
            return [];
        }
        if (res.ok) {
            const items = unwrap(res.data, 'inbound');
            return Array.isArray(items) ? items : [];
        }
        throw await toServiceException('GET', url, res.response);
    }
    async getInbound(id) {
        const url = buildUrl(this.#baseUrl, `/email/2025-03-17/inbound/${encodeURIComponent(id)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.email.getInbound',
                attributes: {
                    id,
                },
            },
        });
        if (res.response.status === 404) {
            return null;
        }
        if (res.ok) {
            return unwrap(res.data, 'inbound');
        }
        throw await toServiceException('GET', url, res.response);
    }
    async listOutbound(addressId) {
        const queryParams = new URLSearchParams();
        if (addressId) {
            queryParams.set('address_id', addressId);
        }
        const queryString = queryParams.toString();
        const url = buildUrl(this.#baseUrl, `/email/2025-03-17/outbound${queryString ? `?${queryString}` : ''}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.email.listOutbound',
                attributes: {
                    ...(addressId && { addressId }),
                },
            },
        });
        if (res.response.status === 404) {
            return [];
        }
        if (res.ok) {
            const items = unwrap(res.data, 'outbound');
            return Array.isArray(items) ? items : [];
        }
        throw await toServiceException('GET', url, res.response);
    }
    async getOutbound(id) {
        const url = buildUrl(this.#baseUrl, `/email/2025-03-17/outbound/${encodeURIComponent(id)}`);
        const signal = AbortSignal.timeout(30_000);
        const res = await this.#adapter.invoke(url, {
            method: 'GET',
            signal,
            telemetry: {
                name: 'agentuity.email.getOutbound',
                attributes: {
                    id,
                },
            },
        });
        if (res.response.status === 404) {
            return null;
        }
        if (res.ok) {
            return unwrap(res.data, 'outbound');
        }
        throw await toServiceException('GET', url, res.response);
    }
}
//# sourceMappingURL=email.js.map