import { z } from 'zod';
import { EmailStorageService } from '@agentuity/core';
import { createServerFetchAdapter } from '@agentuity/server';
import { getCatalystUrl } from '../../../catalyst';
import { defaultProfileName, getDefaultRegion } from '../../../config';
import * as tui from '../../../tui';
export function resolveEmailOrgId(ctx) {
    const orgId = ctx.options.orgId ??
        (process.env.AGENTUITY_CLOUD_ORG_ID || ctx.config?.preferences?.orgId);
    if (!orgId) {
        tui.fatal('Organization ID is required. Use --org-id flag or set AGENTUITY_CLOUD_ORG_ID.');
    }
    return orgId;
}
export const EmailAddressSchema = z.object({
    id: z.string(),
    email: z.string(),
    project_id: z.string().optional(),
    provider: z.string().optional(),
    config: z.record(z.string(), z.unknown()).optional(),
    created_at: z.string(),
    updated_at: z.string().optional(),
});
export function truncate(value, length = 200) {
    if (!value) {
        return '-';
    }
    return value.length > length ? `${value.slice(0, length - 3)}...` : value;
}
export async function createEmailAdapter(ctx, region) {
    const orgId = resolveEmailOrgId(ctx);
    const adapter = createServerFetchAdapter({
        headers: {
            Authorization: `Bearer ${ctx.auth.apiKey}`,
            'x-agentuity-orgid': orgId,
        },
    }, ctx.logger);
    const resolvedRegion = region ?? (await getDefaultRegion(ctx.config?.name ?? defaultProfileName, ctx.config));
    const baseUrl = getCatalystUrl(resolvedRegion);
    return new EmailStorageService(baseUrl, adapter);
}
//# sourceMappingURL=util.js.map