import { z } from 'zod';
import { type APIClient, APIResponseSchema } from '../api.ts';
import { MalwareCheckError } from './util.ts';

export const PackageRefSchema = z.object({
	name: z.string(),
	version: z.string(),
});

export type PackageRef = z.infer<typeof PackageRefSchema>;

export const MalwareFindingSchema = z.object({
	name: z.string(),
	version: z.string(),
	reason: z.string(),
});

export type MalwareFinding = z.infer<typeof MalwareFindingSchema>;

export const MalwareCheckSummarySchema = z.object({
	scanned: z.number(),
	flagged: z.number(),
});

export const MalwareCheckListMetadataSchema = z.object({
	fetchedAt: z.string(),
	count: z.number(),
});

export const MalwareCheckResultSchema = z.object({
	action: z.enum(['allow', 'block']),
	summary: MalwareCheckSummarySchema,
	findings: z.array(MalwareFindingSchema),
	list: MalwareCheckListMetadataSchema.optional(),
	error: z.string().optional(),
});

export const MalwareCheckResponseSchema = APIResponseSchema(MalwareCheckResultSchema);

export type MalwareCheckResult = z.infer<typeof MalwareCheckResultSchema>;

export async function projectDeploymentMalwareCheck(
	client: APIClient,
	deploymentId: string,
	packages: PackageRef[],
	signal?: AbortSignal
): Promise<MalwareCheckResult> {
	const resp = await client.request<z.infer<typeof MalwareCheckResponseSchema>>(
		'POST',
		`/security/2026-01-22/${deploymentId}/malware-check`,
		MalwareCheckResponseSchema,
		{
			ecosystem: 'npm',
			packages,
		},
		undefined,
		signal
	);

	if (!resp.success) {
		throw new MalwareCheckError({
			message: resp.message || 'Malware check request failed',
			deploymentId,
		});
	}

	return resp.data;
}
